<?	// copyright (C) 2005 - Matt Newberry, matt@mattnewberry.net
	// all rights reserved, except as licensed under contract
	
	/*	XmlSimpleBaseClass is the abstract base class of all the classes which load and save to xml files on disk. 
		It's purpose is to provide a default constructor, a save function, and error handling functions for it's 
		descendent classes. It is 'simple' because it uses only a subset of the capabilities of the PHP xml parser. 
		
		Each functional child class must provide the missing _p_cdata($parser, $text) constructor callback function 
		for correct handling of it's own xml tags. The following functions may also be overridden as necessary to 
		provide custom parser functionality (visit the PHP documentation for more info):
		
			function _p_start_element($parser, $element, &$attributes);
			function _p_end_element($parser, $element);
			function _p_pi($parser, $pi);
			function _p_default($parser, $ent);
		
		Data members in descendent classes that have names prefixed with 'xml_' will be automatically saved to 
		disk when the save() function is called. The save() function may be overridden if this default mechanism 
		is not sufficient.
		
		A template for child classes can be found at the bottom of this file.
	*/
	
	class XmlSimpleBaseClass {
		var $XML_READABLE_INPUT_FILE_NOT_FOUND = 8001;
		var $xml_pagetitle = "";
		var $_filename = "";
		var $_path = "";
		var $_tag;
		var $_parse_error = false;
		var $_error_code = 0;
		var $_linkRegex = '{\\b((https?://|mailto:)([\\w/\\#:.?+=&%@!\\-]+?))(?=[.:?\\-]*(?:[^\\w/\\#:.?+=&%@!\\-]|$))}';	// http, https, mailto
		var $_fontRegex = '{@B\[([^\]]+)\]|@BI\[([^\]]+)\]|@I\[([^\]]+)\]}i';	// bold | bold-italic | italic
		
		function XmlSimpleBaseClass($filename) {
			$this->_filename = $filename;
			//$this->_path = getcwd().DIRECTORY_SEPARATOR.$filename;
			$this->_path = realpath($filename);
			if (! is_readable($this->_path)) {
				$this->_parse_error = true;
				$this->_error_code = $this->XML_READABLE_INPUT_FILE_NOT_FOUND;
			}
			else {
				$contents = file_get_contents($this->_path);
				$parser = xml_parser_create();
				xml_parser_set_option($parser, XML_OPTION_CASE_FOLDING, false);
				xml_set_object($parser, $this);	// was &$this
				xml_set_element_handler($parser, '_p_start_element', '_p_end_element');
				xml_set_character_data_handler($parser, '_p_cdata');		// abstract handler function!!
				xml_set_processing_instruction_handler($parser, '_p_pi');
				xml_set_default_handler($parser, '_p_default');
				$this->_parse_error = ! xml_parse($parser, $contents, true);
				$this->_error_code = xml_get_error_code($parser);
				xml_parser_free($parser);
			}
		}
		
		function _p_start_element($parser, $element, &$attributes) {
			$this->_tag = $element;
		}
		
		function _p_end_element($parser, $element) {
			$this->_tag = "";		
		}
		
		function _p_default($parser, $ent) {}
		
		function _p_pi($parser, $pi) {}
		
		function printPageTitle() {
			print htmlspecialchars($this->xml_pagetitle);
		}
		
		/* Possible error return codes:
			0 - XML_ERROR_NONE 
				XML_ERROR_NO_MEMORY 
				XML_ERROR_SYNTAX 
				XML_ERROR_NO_ELEMENTS 
				XML_ERROR_INVALID_TOKEN 
				XML_ERROR_UNCLOSED_TOKEN 
				XML_ERROR_PARTIAL_CHAR 
				XML_ERROR_TAG_MISMATCH 
				XML_ERROR_DUPLICATE_ATTRIBUTE 
				XML_ERROR_JUNK_AFTER_DOC_ELEMENT 
				XML_ERROR_PARAM_ENTITY_REF 
				XML_ERROR_UNDEFINED_ENTITY 
				XML_ERROR_RECURSIVE_ENTITY_REF 
				XML_ERROR_ASYNC_ENTITY 
				XML_ERROR_BAD_CHAR_REF 
				XML_ERROR_BINARY_ENTITY_REF 
				XML_ERROR_ATTRIBUTE_EXTERNAL_ENTITY_REF 
				XML_ERROR_MISPLACED_XML_PI 
				XML_ERROR_UNKNOWN_ENCODING 
				XML_ERROR_INCORRECT_ENCODING 
				XML_ERROR_UNCLOSED_CDATA_SECTION 
				XML_ERROR_EXTERNAL_ENTITY_HANDLING 
				XML_ERR_PI_NOT_STARTED
		*/
		
		function parseFailed() {
			return $this->_parse_error;
		}
		
		function getParseError() {
			return $this->_error_code;
		}
		
		function printParseMessage() {
			if ($this->_error_code == $this->XML_READABLE_INPUT_FILE_NOT_FOUND)
				print("XML READABLE INPUT FILE NOT FOUND");
			else
				print(htmlspecialchars(str_replace("_", " ", xml_error_string($this->_error_code))));
		}
		
		function getFilename() {
			return $this->_filename;
		}
		
		function getPath() {
			return $this->_path;
		}
		
		function htmlFilterSelective($text) {
			$output = htmlspecialchars($text);
			$output = preg_replace($this->_linkRegex, "<a href=\"$1\">$3</a>", $output);
			$output = preg_replace($this->_fontRegex, "<strong>$1<em>$2</em></strong><em>$3</em>", $output);
			return $output;
		}
		
		var $PERSISTENCE_PREFIX = "xml_";
		
		function save() {
			$indent = -1;
			$contents =  "<?xml version=\"1.0\" ?>\r\n";
			$contents .= $this->_s_getClassXml($this, $indent);
			$this->_s_file_write_out($contents);
		}
		
		function _s_getClassXml($c, $indent) {
			++$indent;
			$contents = "";
			for ($i = 0; $i < $indent; ++$i)
				$contents .= "\t";
			$contents .= "<".strtolower(get_class($c)).">\r\n";
			$members = get_object_vars($c);
			foreach ($members as $name => $value) {
				if ($this->_s_cstrpos($name, $this->PERSISTENCE_PREFIX) == 0)
					$contents .= $this->_s_getMemberXml(substr($name, strlen($this->PERSISTENCE_PREFIX)), $value, $indent);
			}
			for ($i = 0; $i < $indent; ++$i)
				$contents .= "\t";
			$contents .= "</".strtolower(get_class($c)).">\r\n";
			return $contents;
		}
		
		function _s_getMemberXml($name, $value, $indent) {
			++$indent;
			$contents = "";
			if (is_object($value)) {
				$contents .= $this->_s_getClassXml($value, $indent);
			}
			else if (is_array($value)) {
				for ($i = 0; $i < $indent; ++$i)
					$contents .= "\t";
				$contents .= "<".strtolower($name).">\r\n";
				foreach ($value as $e)
					$contents .= $this->_s_getMemberXml(substr($name, 0, strlen($name)-1), $e, $indent-1);
				for ($i = 0; $i < $indent; ++$i)
					$contents .= "\t";
				$contents .= "</".strtolower($name).">\r\n";
			}
			else {		// scalar value: simple case where the work is ultimately done
				for ($i = 0; $i < $indent; ++$i)
					$contents .= "\t";
				$contents .= "<".strtolower($name).">".htmlspecialchars($value)."</".strtolower($name).">\r\n";
			}
			return $contents;
		}
		
		function _s_file_write_out($text) {		// file_put_contents() not available in PHP4
			$f = fopen ($this->_path, "w");		// TODO: test is_writable()
			fwrite ($f, $text);
			fclose ($f);
			//chmod($fname, 0666);
		}
		
		function _s_cstrpos($haystack, $needle) {
			$pos = strpos($haystack, $needle);
			if (is_bool($pos))
				return -1;		// not found, like C
			return $pos;
		}
	}
	
		// template for creating child classes
	
	class XmlChildClass extends XmlSimpleBaseClass {
		
		function XmlChildClass($filename) {
			$this->XmlSimpleBaseClass($filename);
			//	... custom construction
		}
		
			// required method
		function _p_cdata($parser, $text) {
			switch ($this->_tag) {
				case "a":
					break;
				case "b":
					break;
				case "c":
					break;
			}
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent));
		}
		
			// optional overrides
		// function _p_start_element($parser, $element, &$attributes) {
		// 	parent::_p_start_element($parser, $element, &$attributes);
		// 	...
		// }
		// function _p_end_element($parser, $element) {
		// 	parent::_p_end_element($parser, $element);
		// 	...
		// }
		// function _p_pi($parser, $pi) {
		//	...
		// }
		// function _p_default($parser, $ent) {
		// 	...
		// }
	}
?>