<?	// copyright (C) 2005 - Matt Newberry, matt@mattnewberry.net
	// all rights reserved, except as licensed under contract
	
	/*  A TextBlock is a container holding one or more paragraghs of text; the opening words of the 
		first paragraph may optionally be stored as a 'headline', which can then be displayed in contrasting html
		style, such as bold or italicized.
	
		Public Interface:
		
				//-- constructor ---------------
			
			function TextBlock($filename);
				- creates a TextBlock from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function printPageTitle();
				- prints an identifying string, html-encoded, for an instance of this class (from XmlSimpleBaseClass)
			
			function printHeadline();
				- print the TextBlock's headline, html-encoded, the first time it is called; thereafter prints nothing.
			
			function printImage();
				- print the filename of the optional image, html-encoded; prints nothing if there is no image
			
			function getImage();
				- gets the filename of the optional image
			
			function moreParagraphs();
				- advance the paragraph pointer to the next paragraph; returns false if no more paragraphs
			
			function printNextParagraph();
				- print the current paragraph, html-encoded
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the TextBlock was successfully parsed from xml, otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function printParseMessage();
				- prints error message from the PHP xml parser, html-encoded (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function setHeadline($headline);
				- set the headline text for the TextBlock
			
			function setImage($image);
				- set the optional image filename for the TextBlock
			
			function addParagraph($paragraph);
				- add a paragraph of text to the TextBlock
			
			function addParagraphs($text);
				- add one or more paragraphs of text, separated by a blank line, to the TextBlock
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
			
			function clear();
				- clear the headline and paragraphs
		
		
		Typical usage in HTML display context:
		
			$textblock = new TextBlock("data/textblock.xml");
			if (! $textblock->parseFailed()) {
				$textblock->printImageTag();
				while ($textblock->moreParagraphs()) {
					$textblock->printHeadline();
					$textblock->printNextParagraph();
				}
			}
		
		
		XML Data Format:
		
			<textblock>
				<pagetitle>TEXT</pagetitle>
				<headline>TEXT</headline>
				<image>TEXT</image>
				<paragraphs>
					<paragraph>TEXT</paragraph>
							.
							.
				</paragraphs>
			</textblock>
	*/
	require_once("XmlSimpleBaseClass.class.php");
	
	class TextBlock extends XmlSimpleBaseClass {
		var $xml_headline = "";
		var $xml_image = "";
		var $xml_paragraphs = array();
		var $_index = -1;
		var $_count = 0;
		var $_next = 0;
		
		function TextBlock($filename) {
			$this->XmlSimpleBaseClass($filename);
			$this->_count = count($this->xml_paragraphs);
		}
		
		function _p_start_element($parser, $element, $attributes) {	// was &$attributes
			parent::_p_start_element($parser, $element, $attributes);	// was &$attributes
			if ($this->_tag == "paragraph") {
				$this->xml_paragraphs[] = "";
				++$this->_index;
			}
		}
		
		function _p_cdata($parser, $text) {
			if ($this->_tag == "headline")
				$this->xml_headline .= $text;
			else if ($this->_tag == "image")
				$this->xml_image .= $text;
			else if ($this->_tag == "paragraph")
				$this->xml_paragraphs[$this->_index] .= $text;
			else if ($this->_tag == "pagetitle")
				$this->xml_pagetitle .= $text;
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent));
		}
		
		function clear() {
			$this->xml_headline = "";
			$this->xml_image = "";
			$this->xml_paragraphs = array();
			$this->_count = 0;
			$this->_next = 0;
		}
		
		function setHeadline($headline) {
			$this->xml_headline = $headline;
		}
		
		function setImage($image) {
			$this->xml_image = $image;
		}
		
		function addParagraph($paragraph) {
			$this->xml_paragraphs = $paragraph;
		}
		
		function addParagraphs($text) {
			while (($index = strpos($text, "\r\n\r\n")) > 0) {
				$this->xml_paragraphs[] = trim(substr($text, 0, $index));
				$text = trim(substr($text, $index+2));
			}
			$this->xml_paragraphs[] = $text;
			$this->_count = count($this->xml_paragraphs);
		}
		
		var $_headline_sent = false;
		
		function printHeadline() {
			if (! $this->_headline_sent)
				print htmlspecialchars(trim($this->xml_headline));
			$this->_headline_sent = true;
		}
		
		function printImage() {
			print htmlspecialchars(trim($this->xml_image));
		}
		
		function getImage() {
			return trim($this->xml_image);
		}
		
		function moreParagraphs() {
			if ($this->_next < $this->_count)
				return true;
			else {
				$this->_next = 0;
				$this->xml_headline_sent = false;
				return false;
			}
		}
		
		function printNextParagraph() {
			print $this->htmlFilterSelective(trim($this->xml_paragraphs[$this->_next++]));
		}
		
		function printNextParagraphForEdit() {
			print htmlspecialchars(trim($this->xml_paragraphs[$this->_next++]));
		}
	}
?>