<?	// copyright (C) 2006 - Matt Newberry, matt@mattnewberry.net
	// ALL RIGHTS RESERVED, except as site-licensed under contract
		
	/*  A News object is a container for Article objects; each Article is a container for paragraphs of text,
		the first of which may have it's first several words, its 'headline', singled out for special emphasis. 
		The ordering of menu items is controlled by the administrator.
		
		Public Interface:
		
				//-- constructor ---------------
			
			function News($filename);
				- creates a News object from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function printPageTitle();
				- prints an identifying string, html-encoded, for an instance of this class (from XmlSimpleBaseClass)
			
			function getNextArticle();
			
			function Article.printID();
			
			function Article.printDate();
				- Month DD, YYYY format
			
			function Article.printTimestamp();
				- Month DD, YYYY HH:MM:SS format
			
			function Article.printHeadline();
			
			function Article.printParagraphs();
			
			function Article.printHeadlineTruncated($maxlen);
			
			function Article.moreParagraphs();
			
			function Article.printNextParagraph();
			
			function Article.printImage();
				- print the filename of the optional image, html-encoded; prints nothing if there is no image
			
			function Article.getImage();
				- gets the filename of the optional image
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the News object was successfully parsed from xml, otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function printParseMessage();
				- prints error message from the PHP xml parser, html-encoded (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function addArticle($headline, $body);
			
			function updateArticle($articleid, $headline, $body);
			
			function getArticle($articleid);
			
			function deleteArticle($articleid);
			
			function promoteArticle($articleid);
				- move the item up one place in the display order
			
			function demoteArticle($menuitemid);
				- move the item down one place in the display order
			
			function Article.setTimestamp($timestamp);
			
			function Article.populate($headline, $body);
			
			function Article.addParagraphs($text);
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
		
		
		Typical usage in HTML display context:
		
			$news = new News("data/news.xml");
			while (($article = $news->getNextArticle()) != null) {
				$article->printHeadline();
				while ($article->moreParagraphs())
					$article->printNextParagraph();
			}
		
		
		XML Data Format:
		
			<news>
				<pagetitle>TEXT</pagetitle>
				<articles>
					<article>
						<headline>TEXT</headline>
						<image>TEXT</image>
						<paragraphs>
							<paragraph>TEXT</paragraph>
										.
										.
						</paragraphs>
					</articles>
						.
						.
				</article>
			</news>
	*/
	require_once("XmlSimpleBaseClass.class.php");
	
	class News extends XmlSimpleBaseClass {
		var $xml_articles = array();		// indexed
		var $_article;
		
		function News($filename) {
			$this->XmlSimpleBaseClass($filename);
			//$this->sortDescending();
		}
		
		function _p_start_element($parser, $element, $attributes) {	// was &$attributes
		 	parent::_p_start_element($parser, $element, $attributes);	// was &$attributes
			switch ($element) {
				case "article":
					$this->_article = new Article();
					break;
				case "paragraph":
					$this->_article->xml_paragraphs[] = "";
					++$this->_article->_pindex;
			}
		}
		
		function _p_cdata($parser, $text) {
			switch ($this->_tag) {
				case "timestamp":
					$this->_article->setTimestamp($text);
					break;
				case "headline":
					$this->_article->xml_headline .= $text;
					break;
				case "image":
					$this->_article->xml_image .= $text;
					break;
				case "paragraph":
					$this->_article->xml_paragraphs[$this->_article->_pindex] .= $text;
					break;
				case "pagetitle":
					$this->xml_pagetitle .= $text;
					break;
			}
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent));
		}
		
		function _p_end_element($parser, $element) {
		 	parent::_p_end_element($parser, $element);
		 	static $count = 0;
			switch ($element) {
				case "article":
					$this->_article->_aindex = $count;
					$this->xml_articles[$count++] = $this->_article;
					break;
				case "articles":
					//krsort($this->xml_articles);
					break;
			}
		}
		
		function getNextArticle() {
			if (list($key, $article) = each($this->xml_articles)) {
				return $article;
			}
			else {
				reset($this->xml_articles);
				return false;
			}
		}
		
		function addArticle($headline, $image, $body) {
			$article = new Article();
			$stamp = time();
			$article->setTimestamp($stamp);
			$article->populate($headline, $image, $body);
			$this->xml_articles[] = $article;
			//krsort($this->xml_articles);
			$this->save();
		}
		
		function updateArticle($articleid, $headline, $image, $body) {
			$this->xml_articles[$articleid]->populate($headline, $image, $body);
			$this->save();
		}
		
		function getArticle($articleid) {
			return $this->xml_articles[$articleid];
		}
		
		function deleteArticle($articleid) {
			unset($this->xml_articles[$articleid]);
			$this->save();
		}
		
		function promoteArticle($articleid) {
			if ($articleid > 0 && $articleid < count($this->xml_articles)) {
				$curr = $this->xml_articles[$articleid];
				$this->xml_articles[$articleid] = $this->xml_articles[$articleid-1];
				$this->xml_articles[$articleid-1] = $curr;
				$this->save();
			}
		}
		
		function demoteArticle($articleid) {
			if ($articleid >= 0 && $articleid < count($this->xml_articles)-1) {
				$curr = $this->xml_articles[$articleid];
				$this->xml_articles[$articleid] = $this->xml_articles[$articleid+1];
				$this->xml_articles[$articleid+1] = $curr;
				$this->save();
			}
		}
	}
	
		// ----- Article class -----
	
	class Article {
		var $_linkRegex = '{\\b((https?://|mailto:)([\\w/\\#:.?+=&%@!\\-]+?))(?=[.:?\\-]*(?:[^\\w/\\#:.?+=&%@!\\-]|$))}';	// http, https, mailto
		var $_fontRegex = '{@B\[([^\]]+)\]|@BI\[([^\]]+)\]|@I\[([^\]]+)\]}i';	// bold | bold-italic | italic
		var $xml_timestamp = "";
		var $xml_headline = "";
		var $xml_image = "";
		var $xml_paragraphs = array();		// indexed
		var $_pindex = -1;	// paragraph counter
		var $_aindex;		// aricle's index in News
		
		function Article() {
			;
		}
		
		function setTimestamp($timestamp) {
			$this->xml_timestamp = $timestamp;
		}
		
		function populate($headline, $image, $body) {
			$this->xml_headline = $headline;
			$this->xml_image = $image;
			$this->addParagraphs($body);
		}
		
		function addParagraphs($text) {
			$this->xml_paragraphs = array();
			while (($index = strpos($text, "\r\n\r\n")) > 0) {
				$this->xml_paragraphs[] = trim(substr($text, 0, $index));
				$text = trim(substr($text, $index+2));
			}
			$this->xml_paragraphs[] = $text;
		}
		
		function printID() {
			print htmlspecialchars(trim($this->_aindex));
		}
		
		function printDate() {
			$date = getdate($this->xml_timestamp);
			$datestring = $date["month"]." ".$date["mday"].", ".$date["year"];
			print htmlspecialchars($datestring);
		}
		
		function printTimestamp() {
			$date = getdate($this->xml_timestamp);
			$datestring = $date["month"]." ".$date["mday"].", ".$date["year"]." ".$date["hours"].":".$date["minutes"].":".$date["seconds"];
			print htmlspecialchars($datestring);
		}
		
		function printHeadline() {
			print htmlspecialchars($this->xml_headline);
		}
		
		function printParagraphs() {
			$text = "";
			foreach ($this->xml_paragraphs as $p)
				$text .= trim($p)."\r\n\r\n";
			print $this->htmlFilterSelective($text);
		}
		
		function printParagraphsForEdit() {
			$text = "";
			foreach ($this->xml_paragraphs as $p)
				$text .= trim($p)."\r\n\r\n";
			print htmlspecialchars($text);
		}
		
		function printHeadlineTruncated($maxlen) {
			print htmlspecialchars(substr($this->xml_headline, 0, $maxlen));
			if (strlen($this->xml_headline) > $maxlen)
				print "...";
		}
		
		function printImage() {
			print htmlspecialchars(trim($this->xml_image));
		}
		
		function getImage() {
			return trim($this->xml_image);
		}
		
		function moreParagraphs() {
			if (current($this->xml_paragraphs))
				return true;
			else {
				reset($this->xml_paragraphs);
				return false;
			}
		}
		
		function printNextParagraph() {
			$p = trim(current($this->xml_paragraphs));
			print $this->htmlFilterSelective($p);
			next($this->xml_paragraphs);
		}
		
		function printNextParagraphForEdit() {
			print htmlspecialchars(trim(current($this->xml_paragraphs)));
			next($this->xml_paragraphs);
		}
		
		function htmlFilterSelective($text) {
			$output = htmlspecialchars($text);
			$output = preg_replace($this->_linkRegex, "<a href=\"$1\">$3</a>", $output);
			$output = preg_replace($this->_fontRegex, "<strong>$1<em>$2</em></strong><em>$3</em>", $output);
			$output = str_replace(Array("\r\n", "\n", "\r"), "<br>", $output);	// convert newlines to html
			return $output;
		}
	}
?>