<?	// copyright (C) 2005 - Matt Newberry, matt@mattnewberry.net
	// ALL RIGHTS RESERVED, except as site-licensed under contract
	
	/*  A Contact is a simple data structure containing fields for a person's name and contact information.
		
		Public Interface:
		
				//-- constructor ---------------
			
			function Contact($filename);
				- creates a Contact object from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function printPageTitle();
				- prints an identifying string, html-encoded, for an instance of this class (from XmlSimpleBaseClass)
				
			function printName($html);
				- prints the contact's name, html-encoded; if $html = true a <br /> will be appended
			
			function printPhone($html);
				- prints the contact's phone number, html-encoded; if $html = true a <br /> will be appended
			
			function printFax($html);
				- prints the contact's fax number, html-encoded; if $html = true a <br /> will be appended
			
			function printAddress($html);
				- prints the contact's address, html-encoded; if $html = true a <br /> will be appended
			
			function printLocation($html);
				- prints the contact's location, html-encoded; if $html = true a <br /> will be appended
			
			function printEmail($html);
				- prints the contact's email address, html-encoded; if $html = true, a full emailhyperlink
					will be printed followed by a <br />
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the Contact was successfully parsed from xml, otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function printParseMessage();
				- prints error message from the PHP xml parser, html-encoded (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
				
			function populate($name, $phone, $fax, $address, $location, $email);
				- populates all fields of the contact
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
		
		
		Typical usage in HTML display context:
		
			$contact = new Contact("data/contact.xml");
			if (! $contact->parseFailed()) {
				Name:    $contact->printName(true);
				Address: $contact->printAddress(true);
				Location $contact->printLocation(true);
				Phone:   $contact->printPhone(true);
				Fax:     $contact->printFax(true);
				Email:   $contact->printEmail(false);
			}
		
		
		XML Data Format:
		
			<contact>
				<pagetitle>TEXT</pagetitle>
				<name>TEXT</name>
				<phone>TEXT</phone>
				<fax>TEXT</fax>
				<address>TEXT</address>
				<location>TEXT</location>
				<email>TEXT</email>
			</contact>
	*/
	require_once("XmlSimpleBaseClass.class.php");
	
	class Contact extends XmlSimpleBaseClass {
		var $xml_name;
		var $xml_phone;
		var $xml_fax;
		var $xml_address;
		var $xml_location;
		var $xml_email;
		
		function Contact($filename) {
			$this->XmlSimpleBaseClass($filename);
		}
		
		function _p_cdata($parser, $text) {
			switch ($this->_tag) {
				case "name":
					$this->xml_name .= $text;
					break;
				case "phone":
					$this->xml_phone .= $text;
					break;
				case "fax":
					$this->xml_fax .= $text;
					break;
				case "address":
					$this->xml_address .= $text;
					break;
				case "location":
					$this->xml_location .= $text;
					break;
				case "email":
					$this->xml_email .= $text;
					break;
				case "pagetitle":
					$this->xml_pagetitle .= $text;
					break;
			}
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent));
		}
		
		function populate($name, $phone, $fax, $address, $location, $email) {
			$this->xml_name = $name;
			$this->xml_phone = $phone;
			$this->xml_fax = $fax;
			$this->xml_address = $address;
			$this->xml_location = $location;
			$this->xml_email = $email;
			$this->save();
		}
		
		function printName($html = false) {
			if (! $html)
				print htmlspecialchars($this->xml_name);
			else if (strlen($this->xml_name) > 0)
				print htmlspecialchars($this->xml_name)."<br />";
		}
		
		function printPhone($html = false) {
			if (! $html)
				print htmlspecialchars($this->xml_phone);
			else if (strlen($this->xml_phone) > 0)
				print htmlspecialchars($this->xml_phone)."<br />";
		}
		
		function printFax($html = false) {
			if (! $html)
				print htmlspecialchars($this->xml_fax);
			else if (strlen($this->xml_fax) > 0)
				print htmlspecialchars($this->xml_fax)."<br />";
		}
		
		function printAddress($html = false) {
			if (! $html)
				print htmlspecialchars($this->xml_address);
			else if (strlen($this->xml_address) > 0)
				print htmlspecialchars($this->xml_address)."<br />";
		}
		
		function printLocation($html = false) {
			if (! $html)
				print htmlspecialchars($this->xml_location);
			else if (strlen($this->xml_location) > 0)
				print htmlspecialchars($this->xml_location)."<br />";
		}
		
		function printEmail($html = false) {
			if (! $html)
				print htmlspecialchars($this->xml_email);
			else if (strlen($this->xml_email) > 0)
				print "<a href=\"mailto:".htmlspecialchars($this->xml_email)."\">".htmlspecialchars($this->xml_email)."</a><br />";
		}
	}
?>