<?	// copyright (C) 2006 - Matt Newberry, matt@mattnewberry.net
	// ALL RIGHTS RESERVED, except as site-licensed under contract
		
	/*  A Journal object is a container for Entry objects; each Entry is a container for paragraphs of text,
		the first of which may have it's first several words, its 'headline', singled out for special emphasis. 
		The Entries are sorted in reverse order by date, but the order may be reversed by the administrator.
		
		Public Interface:
		
				//-- constructor ---------------
			
			function Journal($filename);
				- creates a Journal object from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function printPageTitle();
				- prints an identifying string, html-encoded, for an instance of this class (from XmlSimpleBaseClass)
			
			function sortAscending();
				- sorts latest submissions last
			
			function sortDescending();
				- sorts latest submissions first
			
			function getNextEntry();
			
			function Entry.printID();
			
			function Entry.printDate();
				- Month DD, YYYY format
			
			function Entry.printTimestamp();
				- Month DD, YYYY HH:MM:SS format
			
			function Entry.printHeadline();
			
			function Entry.printParagraphs();
			
			function Entry.printHeadlineTruncated($maxlen);
			
			function Entry.moreParagraphs();
			
			function Entry.printNextParagraph();
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the Journal object was successfully parsed from xml, otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function printParseMessage();
				- prints error message from the PHP xml parser, html-encoded (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function addEntry($headline, $body);
			
			function updateEntry($entryid, $headline, $body);
			
			function getEntry($entryid);
			
			function deleteEntry($entryid);
			
			function Entry.setTimestamp($timestamp);
			
			function Entry.populate($headline, $body);
			
			function Entry.addParagraphs($text);
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
		
		
		Typical usage in HTML display context:
		
			$journal = new Journal("data/journal.xml");
			while (($entry = $journal->getNextEntry()) != null) {
				$entry->printHeadline();
				while ($entry->moreParagraphs())
					$entry->printNextParagraph();
			}
		
		
		XML Data Format:
		
			<journal>
				<pagetitle>TEXT</pagetitle>
				<entries>
					<entry>
						<timestamp>NUMBER</timestamp>
						<headline>TEXT</headline>
						<paragraphs>
							<paragraph>TEXT</paragraph>
										.
										.
						</paragraphs>
					</entry>
						.
						.
				</entries>
			</journal>
	*/
	require_once("XmlSimpleBaseClass.class.php");
	
	class Journal extends XmlSimpleBaseClass {
		var $xml_entries = array();		// associative; key = submission timestamp
		var $_entry;
		
		function Journal($filename) {
			$this->XmlSimpleBaseClass($filename);
			$this->sortDescending();
		}
		
		function _p_start_element($parser, $element, $attributes) {	// was &$attributes
		 	parent::_p_start_element($parser, $element, $attributes);	// was &$attributes
			switch ($element) {
				case "entry":
					$this->_entry = new Entry();
					break;
				case "paragraph":
					$this->_entry->xml_paragraphs[] = "";
					++$this->_entry->_index;
			}
		}
		
		function _p_cdata($parser, $text) {
			switch ($this->_tag) {
				case "timestamp":
					$this->_entry->setTimestamp($text);
					break;
				case "headline":
					$this->_entry->xml_headline .= $text;
					break;
				case "paragraph":
					$this->_entry->xml_paragraphs[$this->_entry->_index] .= $text;
					break;
				case "pagetitle":
					$this->xml_pagetitle .= $text;
					break;
			}
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent));
		}
		
		function _p_end_element($parser, $element) {
		 	parent::_p_end_element($parser, $element);
			switch ($element) {
				case "entry":
					$this->xml_entries[$this->_entry->xml_timestamp] = $this->_entry;
					break;
				case "entries":
					krsort($this->xml_entries);
					break;
			}
		}
		
		function getNextEntry() {
			if (list($key, $entry) = each($this->xml_entries)) {
				return $entry;
			}
			else {
				reset($this->xml_entries);
				return false;
			}
		}
		
		function addEntry($headline, $body) {
			$entry = new Entry();
			$stamp = time();
			$entry->setTimestamp($stamp);
			$entry->populate($headline, $body);
			$this->xml_entries[$stamp] = $entry;
			krsort($this->xml_entries);
			$this->save();
		}
		
		function updateEntry($entryid, $headline, $body) {
			$entry =& $this->getEntry($entryid);	// byref
			$entry->populate($headline, $body);
			$this->save();
		}
		
		function getEntry($entryid) {
			return $this->xml_entries[$entryid];
		}
		
		function deleteEntry($entryid) {
			foreach ($this->xml_entries as $a) {
				if ($a->xml_timestamp == $entryid)
					unset($this->xml_entries[$entryid]);
			}
			krsort($this->xml_entries);
			$this->save();
		}
		
		function sortAscending() {
			ksort($this->xml_entries);
		}
		
		function sortDescending() {
			krsort($this->xml_entries);
		}
	}
	
		// ----- Entry class -----
	
	class Entry {
		var $_linkRegex = '{\\b(([^"]https?://|[^"]mailto:)([\\w/\\#:.?+=&%@!\\-]+?))(?=[.:?\\-]*(?:[^\\w/\\#:.?+=&%@!\\-]|$))}';	// http, https, mailto
		var $_fontRegex = '{@B\[([^\]]+)\]|@BI\[([^\]]+)\]|@I\[([^\]]+)\]}i';	// bold | bold-italic | italic
		var $_anchorRegex = '/(@A\[)([^\s]+)(\s)([^\]]+)(\])/i';		// an anchor with a link and a label
		var $xml_timestamp = "";
		var $xml_headline = "";
		var $xml_paragraphs = array();		// indexed
		var $_index = -1;
		
		function Entry() {
			;
		}
		
		function setTimestamp($timestamp) {
			$this->xml_timestamp = $timestamp;
		}
		
		function populate($headline, $body) {
			$this->xml_headline = $headline;
			$this->addParagraphs($body);
		}
		
		function addParagraphs($text) {
			$this->xml_paragraphs = array();
			while (($index = strpos($text, "\r\n\r\n")) > 0) {
				$this->xml_paragraphs[] = trim(substr($text, 0, $index));
				$text = trim(substr($text, $index+2));
			}
			$this->xml_paragraphs[] = $text;
		}
		
		function printID() {
			print htmlspecialchars(trim($this->xml_timestamp));
		}
		
		function printDate() {
			$date = getdate($this->xml_timestamp);
			$datestring = $date["month"]." ".$date["mday"].", ".$date["year"];
			print htmlspecialchars($datestring);
		}
		
		function printTimestamp() {
			$date = getdate($this->xml_timestamp);
			$datestring = $date["month"]." ".$date["mday"].", ".$date["year"]." ".$date["hours"].":".$date["minutes"].":".$date["seconds"];
			print htmlspecialchars($datestring);
		}
		
		function printHeadline() {
			print htmlspecialchars(trim($this->xml_headline));
		}
		
		function printParagraphs() {
			$text = "";
			foreach ($this->xml_paragraphs as $p)
				$text .= trim($p)."\r\n\r\n";
			print $this->htmlFilterSelective($text);
		}
		
		function printParagraphsForEdit() {
			$text = "";
			foreach ($this->xml_paragraphs as $p)
				$text .= trim($p)."\r\n\r\n";
			print htmlspecialchars($text);
		}
		
		function printHeadlineTruncated($maxlen) {
			print htmlspecialchars(substr($this->xml_headline, 0, $maxlen));
			if (strlen($this->xml_headline) > $maxlen)
				print "...";
		}
		
		function moreParagraphs() {
			if (current($this->xml_paragraphs))
				return true;
			else {
				reset($this->xml_paragraphs);
				return false;
			}
		}
		
		function printNextParagraph() {
			print $this->htmlFilterSelective(trim(current($this->xml_paragraphs)));
			next($this->xml_paragraphs);
		}
		
		function printNextParagraphForEdit() {
			print htmlspecialchars(trim(current($this->xml_paragraphs)));
			next($this->xml_paragraphs);
		}
		
		function htmlFilterSelective($text) {
			$output = htmlspecialchars($text);
			$output = preg_replace($this->_anchorRegex, "<a href=\"$2\">$4</a>", $output);
			$output = preg_replace($this->_linkRegex, "<a href=\"$1\">$3</a>", $output);
			$output = preg_replace($this->_fontRegex, "<strong>$1<em>$2</em></strong><em>$3</em>", $output);
			$output = str_replace(Array("\r\n", "\n", "\r"), "<br>", $output);	// convert newlines to html
			return $output;
		}
	}
?>