<?	/*
		Secure Login System
		-------------------
		This login page implements an encrypted login scheme without requiring SSL. Two cooperating objects 
		are involved, one on the server implemented in PHP, and the other in the browser implemented in Javascript. 
		As long as Javascript is enabled in the browser, the password is never transmitted in clear text.
		
		In the server-side data storage, login passwords are maintained in a SHA1-hashed state; in effect this is a 
		one-way encryption: passwords can be encrypted but never decrypted. When someone logs in, his submitted 
		password must be encrypted by the same method, then compared to the stored password.
		
		Implementation
		--------------
		When the user requests the login page (login.php) a PHP object of class SecuredLogin is created on the server 
		and stored in the user's session. This object's GetWildcard() method is called to generate a long 
		random string of data for one-time use. The object keeps a copy of the data for itself while passing a 
		copy to the php script. The script in turn sends this text to the user's browser as a Javascript variable 
		called 'wildcard'. Every time the login page is requested, even if reloaded by the same user, a brand new 
		wildcard is generated.
		
		In the browser, the wildcard variable is used to create  a Javascript object of class SecuredLogin. When the user 
		fills in his username and password and submits the login form, the onsubmit event handler calls the Secure() function
		of the SecuredLogin object and passes the entire form contents to it, along with the names of the form fields that hold 
		the cleartext password and a placeholder for the hashed password. This function does several things: 
		
			- it retrieves the value of the cleartext password field (that is, the password the user typed in) from the form
			- it creates a SHA1 hash of the password, concatenates that hash with the stored wildcard, and then 
				hashes the concatenation
			- this final hash is added to the form as a hidden field; schematically it's value looks like 
				this: (hash(wildcard+hash(password)))
			- finally the unencrypted password is deleted from the form and the form is submitted. 
		
		The following three form fields are submitted to the server:
		
			username = "username_as_entered_by_user"
			password = ""	(blank!)
			hash = "a_long_string_of_random_characters"
		
		Upon receiving the login request, the server script does several things which parallel the steps taken 
		by the browser:
		
			- using the submitted username, it looks up the encrypted password for that user from data storage
			- it retrieves the SecuredLogin object, created when the form was requested, from session storage
			- it passes the looked-up password and the submitted 'hash' field to the Approve() method of 
				the SecuredLogin object, which remembers the value of the generated wildcard string
			- the SecuredLogin object concatenates the wildcard with the looked-up password (already hashed) and 
				hashes the concatenation, producing the following: (hash(wildcard+hashed_password))
			- SecuredLogin compares its generated hash to the 'hash' field received from the browser; if they are 
				identical, the SecuredLogin.Approve() method returns true, and the user is logged in.
				
		If Javascript is not enabled in the user's browser, the 'hash' field will be empty and, after a warning to the user,
		the password will be sent in clear text. Rather than reject the login, since the clear password has already been sent 
		(that is, the damage, if any, has already been done), an attempt is made to validate the user using the clear password 
		and the SecuredLogin.ApproveClear() function.
		
		Benefits
		--------
		The password is not transmitted in clear text. Furthermore, the hashed password is never transmitted 
		either, as this would enable replay attacks if intercepted; instead the hashed password is rehashed 
		with a one-time wildcard, which is useless for future login attempts, and this double-hashed value is 
		what is transmitted.
		
		Weaknesses
		----------
		On the server, passwords are stored hashed, but not 'salted'. (Salting is adding a random, but consistent, 
		value [the 'salt'] to each password before it is hashed.) Unsalted passwords are subject to a dictionary 
		attack in which the stored passwords are compared, not to a list of common passwords, but to a hashed list 
		of common passwords. If two hashes are found to match, the unhashed value (the actual password!) is already 
		known to the attacker.
		
		An augmenting weakness in this system is that the hashed passwords are stored in a data file which is 
		http-readable if discovered by an attacker: a ready-made list for the dictionary attack.
		
		The use of salted passwords is apparently precluded in this application by the need to keep the salt value 
		secret: the salt value cannot be transmitted to the browser for inclusion in the browser-generated hash. 
		It may be possible to work around these limitations, at the cost of additional complexity in the login system.
	*/
	
	ini_set("display_errors", "1");
	require("../include/SecuredLogin.class.php");
	require("../include/Accounts.class.php");
	require("../include/edit_levels.php");
	
	session_start();
	if (! isset($_SESSION["SecuredLogin"]) || $_SESSION["SecuredLogin"] == null) {
		$_SESSION["SecuredLogin"] = new SecuredLogin();
	}
	$security = $_SESSION["SecuredLogin"];
	
	$logins = new Accounts("data/accounts.xml");
	if ($logins->parseFailed()) {
		exit();
	}
	
	$method = strtolower($_SERVER["REQUEST_METHOD"]);
	if ($method == "get") {				// show login screen
		$wildcard = $security->GetWildcard();
		$_SESSION["SecuredLogin"] = $security;	// re-save with wildcard data
	}
	else if ($method == "post") {		// process login attempt
		$submittedUsername = $_POST["username"];
		$lookedUpPassword = LookupPassword($submittedUsername, $logins);
		if ($_POST["hash"] != "") {					// password sent hashed
			$submittedPassword = $_POST["hash"];
			if ($security->Approve($submittedPassword, $lookedUpPassword)) {
				$_SESSION["logged_in"] = true;		// passed!
				$_SESSION["access_level"] = LookupLevel($_POST["username"], $logins);
				$_SESSION["page_levels"] = $pageLevels;
				header("Location: .");				// redirect to admin menu
				return;
			}
		}
		else {		// evidently Javascript disabled...
			$submittedPassword = $_POST["password"];// ... so password sent in the clear
			if ($security->ApproveClear($submittedPassword, $lookedUpPassword)) {
				$_SESSION["logged_in"] = true;		// passed!
				$_SESSION["access_level"] = LookupLevel($_POST["username"], $logins);
				$_SESSION["page_levels"] = $pageLevels;
				header("Location: .");				// redirect to admin menu
				return;
			}
		}
			// login failed
		$_SESSION["logged_in"] = false;		// revoke previous login, if any
		$_SESSION["access_level"] = -1;
		$_SESSION["page_levels"] = null;
		header("Location: loginfailure.html");
	}
	
	function LookupPassword($user, $logins) {
		$login = $logins->getLogin($user);
		if ($login != null)
			return $login->getPassword();
		else
			return '';
	}
	
	function LookupLevel($user, $logins) {
		$login = $logins->getLogin($user);
		if ($login != null)
			return $login->getLevel();
		else
			return '';
	}
?>

<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN"
	"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">

<html>
<head>
	<title>Login</title>
	<link href="admin.css" rel="stylesheet" type="text/css" />
	<script src="../include/SecuredLogin.class.js" language="JavaScript1.1" type="text/javascript"></script>
	<script type="text/javascript" language="JavaScript1.1">
		function formRequire(frm, usernameFieldname, passwordFieldname) {
			if (frm[usernameFieldname].value == '' || frm[passwordFieldname].value == '') {
				alert('Please enter a Username and a Password.');
				return false;
			}
			return true;
		}
		var wildcard = "<? print($wildcard) ?>"
		var security = new SecuredLogin(wildcard);
	</script>
</head>
<body>
<div align="center">
	<h2>Administrator Login</h2>
	<form id="loginform" name="loginform" method="POST" onsubmit="return security.Secure(this, 'password', 'hash')">
		<input type="hidden" name="hash" />
		<table>
		<tr><th>Username</th><td><input type="text" name="username" value="" /></td></tr>
		<tr><th>Password</th><td><input type="password" name="password" value="" /></td></tr>
		<tr><td>&nbsp;</td></tr>
		<tr><td class="submit" colspan="2"><input type="submit" value="Log In" onclick="return formRequire(document.loginform, 'username', 'password')" /></td></tr>
		</table>
	</form>
	<br />
	<noscript>Javascript is disabled - your password will not be encrypted!</noscript>
</div>
</body>
</html>
