<?	/*
		Secure Login System
		-------------------
		This login page implements an encrypted login scheme without requiring SSL. Two cooperating objects 
		are involved, one on the server implemented in PHP, and the other in the browser implemented in Javascript. 
		As long as Javascript is enabled in the browser, the password is never transmitted in clear text.
		
		In the server-side data storage, login passwords are maintained in a SHA1-hashed state; in effect this is a 
		one-way encryption: passwords can be encrypted but never decrypted. When someone logs in, his submitted 
		password must be encrypted by the same method, then compared to the stored password.
		
		Implementation
		--------------
		When the user requests the login page (login.php) an object of class AuthSHA1 is created on the server 
		and stored in the user's session. This object's GetChallengeText() method is called to generate a long 
		random string of data for one-time use. The object keeps a copy of the data for itself while passing a 
		copy to the php script. The script in turn sends this text to the user's browser as a hidden form field 
		called 'wildcard'. Every time the login page is requested, even if reloaded by the same user, a brand new 
		wildcard is generated.
		
		When the user fills in his username and password and submits the login form, the onclick event handler 
		creates an object of class LoginSHA1, and passes the entire form contents to it. This object does several things: 
		
			- it retrieves the values of the 'username', 'password' and 'wildcard' fields from the form
			- it creates a SHA1 hash of the password, concatenates that hash with the wildcard, and then hashes 
				the concatenation
			- this final hash is added to the form as a hidden field called 'hash'; schematically it looks like 
				this: (hash(wildcard+hash(password)))
			- finally the unencrypted password is deleted from the form and the form is submitted. 
		
		The following four form fields are submitted to the server:
		
			username = "username_as_entered_by_user"
			password = ""	(blank!)
			wildcard = "one_long_string_of_random_characters"
			hash = "another_long_string_of_random_characters"
		
		Upon receiving the login request, the server script does several things which parallel the steps taken 
		by the browser:
		
			- using the submitted username, it looks up the encrypted password for that user from data storage
			- it retrieves the AuthSHA1 object, created when the form was requested, from session storage
			- it passes the looked-up password and the submitted 'hash' field to the Authenticate() method of 
				the AuthSHA1 object, which remembers the value of the generated wildcard string
			- the AuthSHA1 object concatenates the wildcard with the looked-up password (already hashed) and 
				hashes the concatenation, producing the following: (hash(wildcard+hashed_password))
			- AuthSHA1 compares its generated hash to the 'hash' field received from the browser; if they are 
				identical, the Authenticate() method returns true, and the user is logged in.
		
		Benefits
		--------
		The password is never transmitted in clear text. Furthermore, the hashed password is never transmitted 
		either, as this would enable replay attacks if intercepted; instead the hashed password is rehashed 
		with a one-time wildcard, which is useless for future login attempts, and this double-hashed value is 
		what is transmitted.
		
		Weaknesses
		----------
		On the server, passwords are stored hashed, but not 'salted'. (Salting is adding a random, but consistent, 
		value [the 'salt'] to each password before it is hashed.) Unsalted passwords are subject to a dictionary 
		attack in which the stored passwords are compared, not to a list of common passwords, but to a hashed list 
		of common passwords. If two hashes are found to match, the unhashed value (the actual password!) is already 
		known to the attacker.
		
		An augmenting weakness in this system is that the hashed passwords are stored in a data file which is 
		http-readable if discovered by an attacker: a ready-made list for the dictionary attack.
		
		The use of salted passwords is apparently precluded in this application by the need to keep the salt value 
		secret: the salt value cannot be transmitted to the browser for inclusion in the browser-generated hash. 
		It may be possible to work around these limitations, at the cost of additional complexity in the login system.
	*/
	
	ini_set("display_errors", "1");
	require("../include/Auth.class.php");
	require("../include/SecuredLogin.class.php");
	require("../include/Accounts.class.php");
	require("../include/edit_levels.php");
	session_start();
	
	$logins = new Accounts("data/accounts.xml");
	if ($logins->parseFailed()) {
		exit();
	}
	
	$method = strtolower($_SERVER["REQUEST_METHOD"]);
	if ($method == "get") {				// show login screen
		$auth = new AuthSHA1();
		//$auth->SetMaxTries(3);
		$wildcard = $auth->GetChallengeText();
		$_SESSION["auth"] = $auth;
	}
	else if ($method == "post") {		// process login attempt
		if (! isset($_SESSION["auth"])) {					// not logged in
			header("Location: index.php");
			return;
		}
		$auth = $_SESSION["auth"];
		$password = LookupPassword($_POST["username"], $logins);
		if ($auth->Authenticate($password, $_POST["hash"])) {
			$_SESSION["logged_in"] = true;		// passed!
			$_SESSION["access_level"] = LookupLevel($_POST["username"], $logins);
			$_SESSION["page_levels"] = $pageLevels;
			header("Location: .");				// redirect to admin menu
			return;
		}
		else {									// login failed
			$_SESSION["logged_in"] = false;		// revoke previous login, if any
			$_SESSION["access_level"] = -1;
			$_SESSION["page_levels"] = null;
			header("Location: loginfailure.html");
			return;
		}
		header("Location: loginfailure.html");	// failsafe: should never fall through to here
	}
	
	function LookupPassword($user, $logins) {
		$login = $logins->getLogin($user);
		if ($login != null)
			return $login->getPassword();
		else
			return '';
	}
	
	function LookupLevel($user, $logins) {
		$login = $logins->getLogin($user);
		if ($login != null)
			return $login->getLevel();
		else
			return '';
	}
?>

<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN"
	"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">

<html>
<head>
	<title>Login</title>
	<link href="admin.css" rel="stylesheet" type="text/css" />
	<script src="../include/formverify.js" language="JavaScript1.1" type="text/javascript"></script>
	<script src="../include/Auth.class.js" language="JavaScript1.1" type="text/javascript"></script>
	<script src="../include/SecuredLogin.class.js" language="JavaScript1.1" type="text/javascript"></script>
</head>
<body>
<div align="center">
	<h2>Administrator Login</h2>
	<form name="login" method="POST" onsubmit="
	    this.username.required = true;
	    return verify(this);
	">
		<input type="hidden" name="wildcard" value="<? echo htmlspecialchars($wildcard) ?>" />
		<input type="hidden" name="hash" />
		<table>
		<tr><th>Username</th><td><input type="text" name="username" value="" /></td></tr>
		<tr><th>Password</th><td><input type="password" name="password" value="" /></td></tr>
		<tr><td>&nbsp;</td></tr>
		<tr><td class="submit" colspan="2"><input type="submit" value="Log In" onclick="return LoginSHA1(document.login)" /></td></tr>
		</table>
	</form>
</div>	
</body>
</html>
