<?	// copyright (C) 2005 - Matt Newberry, matt@mattnewberry.net
	// ALL RIGHTS RESERVED, except as site-licensed under contract
	
	/*  A Properties object is a simple data structure that holds any number of static name/value pairs. There is 
		no admin interface for adding, editing or deleting values: the static values are read from an xml data file 
		and are then available to be inserted into html output with the printValue($name) function, or to be obtained 
		for other purposes with the getValue($name) function. New display variables can be easily added to the data
		file as:
		
			<name>value</name>
		
		then as easily displayed in the output html file by calling printValue("name").
		
		A save function is provided to override the inherited, but inappropriate, save(); it is not needed and
		not used, but provided in anticipation of an editable Properties object.
		
		Public Interface:
		
				//-- constructor ---------------
			
			function Properties($filename);
				- creates a Properties object from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function printPageTitle();
				- prints an identifying string, html-encoded, for an instance of this class (from XmlSimpleBaseClass)
			
			function printValue($name);
				- prints the value of the given name, html-encoded
			
			function getValue($name);
				- returns the string value of the given name, not encoded
			
			function updateValue($name, $value);
				- updates the value of the named entry
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the Properties object was successfully parsed from xml, 
					otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function printParseMessage();
				- prints error message from the PHP xml parser, html-encoded (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function save();
				- save data back to disk under original filename (overrides XmlSimpleBaseClass)
		
		
		Typical usage in HTML display context:
		
			$properties = new Properties("data/properties.xml");
			if (! $properties->parseFailed))
				$properties.printValue("FirstName");
		
		
		XML Data Format:
		
			<properties>
				<pagetitle>TEXT</pagetitle>
				<NAME>VALUE</NAME>
						.
						.
			</properties>
	*/
	require_once("XmlSimpleBaseClass.class.php");
	
	class Properties extends XmlSimpleBaseClass {
		var $xml_properties = array();
		
		function Properties($filename) {
			$this->XmlSimpleBaseClass($filename);
			if (array_key_exists("pagetitle", $this->xml_properties))
				$this->xml_pagetitle = $this->xml_properties["pagetitle"];
		}
		
		 function _p_start_element($parser, $element, $attributes) {	// was &$attributes
		 	parent::_p_start_element($parser, $element, $attributes);	// was &$attributes
		 	if ($element != "properties") {
				$this->xml_properties[$element] = "";
			}
		 }
		
		function _p_cdata($parser, $text) {
		 	if ($this->_tag != "properties" && trim($this->_tag) != "") {
				$this->xml_properties[$this->_tag] .= $text;
			}
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent,ENT_QUOTES,"ISO-8859-1"));
		}
		
		function getValue($name) {
			if (array_key_exists($name, $this->xml_properties))
				return trim($this->xml_properties[$name]);
			else
				return null;
		}
		
		function updateValue($name, $value) {
			if (array_key_exists($name, $this->xml_properties))
				$this->xml_properties[$name] = $value;
		}
		
		function printValue($name) {
			if (array_key_exists($name, $this->xml_properties))
				print htmlentities(trim($this->xml_properties[$name]),ENT_QUOTES,"ISO-8859-1");
		}
		
		function save() {
			$contents =  "<?xml version=\"1.0\" encoding=\"ISO-8859-1\" standalone=\"no\" ?>\r\n<!DOCTYPE news SYSTEM \"http://www.uuism.net/uutemplate/demo.dtd\">\r\n";
			$contents .= "<properties>\r\n";
			foreach ($this->xml_properties as $key => $val)
				$contents .= "\t<".$key.">".htmlentities(trim($val),ENT_QUOTES,"ISO-8859-1")."</".$key.">\r\n";
			$contents .= "</properties>\r\n";
			$this->_s_file_write_out($contents);
		}
	}
?>
