<?
/** 
 * Provides an interface to access a directory listing and access information about the files
 * in that directory
 *
 * @author Dan Pupius
 * @version 1.0
 * @copyright 2004 pupius.co.uk  <http://pupius.co.uk>
 * @package Utility
 * @subpackage Classes
 */
 
 /* extensions as noted by Matt Newberry */
 
class DirectoryListing {

	/**
	 * @var string The path that the current directory listing represents
	 */
	var $path = "";
		

	/**
	 * @var array Array of directory objects (note not a recursive directory listing).  Includes names and minimal information
	 */
	var $directories = array();
	
	/**
	 * @var array Array of file information bjects
	 */
	var $files = array();



	/** 
	 * Constructor function, sets up the object based on the path
	 *
	 * @param string $path Path of directory to open
	 * @access public
	 * @return boolean Whether or not directory list was successfully generated
	 */
	function DirectoryListing($path) {
		if(!is_dir($path)) return false;
		
		$path = rtrim($path,"/");
		
		$this->path = $path;
		
		//loop through contents of directory and construct arrays of files and directories
		if($handle = opendir($path)) {

			//loop through each file in the directory and get information on it (not all of it will be needed, but it's added just in case)
			while (false !== ($f = readdir($handle))) {
				//directory listing only ignores . and .. files, whatever uses this class should handle security of hidden files
				if($f != "." && $f != "..") {
					//get the full path for current file/dir
					$fullpath = "$path/$f";

					//if its a directory then push a directory object onto the array
					if(is_dir($fullpath)) array_push($this->directories,new DirectoryInfo($fullpath));

					//if its a file then push a file object onto the array
					else if(is_file($fullpath)) array_push($this->files,new FileInfo($fullpath));
				}
			}
			closedir($handle);
			
			return true;
			
		} else return false;
	}
	
	/** 
	 * Returns the number of files in the current directory
	 *
	 * @return integer Number of files in current directory
	 * @access public
	 */
	function num_files() {
		return count($this->files);
	}
	
	/** 
	 * Returns the number of directories in the current directory
	 *
	 * @return integer Number of directories in current directory
	 * @access public
	 */	
	function num_directories() {
		return count($this->directories);
	}
	
	
	/** 
	 * Sort the file array
	 *
	 * @param boolean $asc Whether to sort ascending or descending
	 * @param string $field What property of the FileInfo object to use as the sortable field
	 * @param boolean $natlang Whether to use a natural language based sort	 
	 * @access public
	 */
	function sort_files($asc=true,$field="filename",$natlang=true) {
		$GLOBALS["file_sort_asc"] = $asc;
		$GLOBALS["file_sort_field"] = $field;
		$GLOBALS["file_sort_natlang"] = $natlang;
		usort($this->files,array("DirectoryListing","filesort"));
	}
	
	/** 
	 * Eliminate files not matching one of the extensions
	 *
	 * @param array $extarray Array of file extensions to keep
	 * @access public
	 * @author Matt Newberry
	 */
	function filter_by_extensions($extarray) {
		$filtered = Array();
		foreach ($this->files as $f) {
			foreach ($extarray as $e) {
				if (strtolower($f->ext) == strtolower($e))
					array_push($filtered, $f);
			}
		}
		$this->files = $filtered;
	}
	
	
	/** 
	 * Return files sequentially until used up, then return false and reset
	 *
	 * @access public
	 * @author Matt Newberry
	 */
	function getNextFile() {
		if ($f = current($this->files)) {
			next($this->files);
			return $f;
		}
		else {
			reset($this->files);
			return false;
		}
	}
	
	
	/** 
	 * Sort the directory array
	 *
	 * @param boolean $asc Whether to sort ascending or descending
	 * @param string $field What property of the DirectoryInfo object to use as the sortable field
	 * @param boolean $natlang Whether to use a natural language based sort	 
	 * @access public
	 */
	function sort_directories($asc=true,$field="dirname",$natlang=true) {
		$GLOBALS["file_sort_asc"] = $asc;
		$GLOBALS["file_sort_field"] = $field;
		$GLOBALS["file_sort_natlang"] = $natlang;
		usort($this->directories,array("DirectoryListing","filesort"));
	}
	
	
	
	/** 
	 * File sorting function
	 *
	 * @param mixes $a First item to compare
	 * @param mixes $b Second item to compare
	 * @return integer
	 * @access public
	 */
	function filesort($a,$b) {
		$field = $GLOBALS["file_sort_field"];
		$asc = $GLOBALS["file_sort_asc"];
		$natlang = $GLOBALS["file_sort_natlang"];
		
		$item1 = strtolower($a->$field);
		$item2 = strtolower($b->$field);
	
		if($natlang) $retval = strnatcmp($item1,$item2);
		else if($item1==$item2) $retval =  0;
		else $retval = ($item1 < $item2) ? -1 : 1;
	
		if($asc || $retval==0) return $retval;
		else if($retval==1) return -1;
		else return 1;
	}

}


/** 
 * Creates an object with easy to access information about a partcular file
 *
 * @author Dan Pupius
 * @version 1.0
 * @copyright 2004 pupius.co.uk  <http://pupius.co.uk>
 * @package Utility
 * @subpackage Classes
 */
class FileInfo {

	/**
	 * @var string The filename+extension that the current file object represents
	 */
	var $fullname = "";		// MN

	/**
	 * @var string The path that the current file object represents
	 */
	var $path = "";


	/**
	 * @var string The full path of the file
	 */
	var $realpath = "";

	/**
	 * @var string Filename, without extension
	 */
	var $filename = "";
	
	/**
	 * @var string File's extension, without the dot
	 */
	var $ext = "";
	
	/**
	 * @var string Directory location of the file
	 */
	var $dir = "";
	
	/**
	 * @var string Full directory path of the file
	 */
	var $realdir = "";
	
	/**
	 * @var integer Raw size of the file
	 */
	var $size = "";
	
	/**
	 * @var integer Raw timestamp of last modified date
	 */
	var $modified = "";
	

	/** 
	 * Constructor function, sets up the object based on the path
	 *
	 * @param string $path Path of file to open
	 * @access public
	 * @return boolean Whether or not file object was successfully generated
	 */
	function FileInfo($path) {
		if(!file_exists($path) || !is_file($path)) return false;
		
		$this->path = $path;
		$this->realpath = realpath($path);
		
		$pathinfo = pathinfo($path);
		$this->ext = $pathinfo["extension"];
		$this->dir = $pathinfo["dirname"];
		$this->realdir = realpath($pathinfo["dirname"]);
		$this->fullname = $pathinfo["basename"];
		$this->filename = str_replace(".".$this->ext,"",$pathinfo["basename"]);
		$this->size = filesize($path);	
		$this->modified =  filemtime($path);
		
		return true;
	}
}


/** 
 * Creates an object with easy to access information about a partcular directory
 *
 * @author Dan Pupius
 * @version 1.0
 * @copyright 2004 pupius.co.uk  <http://pupius.co.uk>
 * @package Utility
 * @subpackage Classes
 */
class DirectoryInfo {


	/**
	 * @var string The path that the current file object represents
	 */
	var $path = "";


	/**
	 * @var string The full path of the file
	 */
	var $realpath = "";

	/**
	 * @var string Direcotry name
	 */
	var $dirname = "";
	

	/** 
	 * Constructor function, sets up the object based on the path
	 *
	 * @param string $path Path of file to open
	 * @access public
	 * @return boolean Whether or not file object was successfully generated
	 */
	function DirectoryInfo($path) {
		if(!file_exists($path) || !is_dir($path)) return false;
		
		$this->path = $path;
		$this->realpath = realpath($path);
		
		$p = explode("/",$path);
		$this->dirname = array_pop($p);
		
		return true;
	}
}

?>
