<?	// copyright (C) 2005 - Matt Newberry, matt@mattnewberry.net
	// all rights reserved, except as licensed under contract
	
	/*  A DefinitionList is a container for an array of Element objects, suitable for display in an
		html definition list. Each element holds a term and its definition. The ordering of elements is 
		controlled by the administrator.
	
		Public Interface:
		
				//-- constructor ---------------
			
			function DefinitionList($filename)
				- creates a DefinitionList from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function printPageTitle();
				- prints an identifying string, html-encoded, for an instance of this class (from XmlSimpleBaseClass)
			
			function getNextElement();
				- returns the next Element sequentially; returns null if no more Elements
			
			function Element.printID();
				- prints the Element's unique identifier
			
			function Element.printTerm();
				- prints the Element's term, html-encoded
			
			function Element.printDefinition();
				- prints the Element's definition, html-encoded
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the DefinitionList was successfully parsed from xml, otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function printParseMessage();
				- prints error message from the PHP xml parser, html-encoded (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function getElement($elementid);
			
			function addElement($term, $definition);
			
			function updateElement($elementid, $term, $definition);
			
			function deleteElement($elementid);
			
			function promoteElement($elementid);
				- move the element up one place in the display order
			
			function demoteElement($elementid);
				- move the element down one place in the display order
			
			function Element.populate($term, $definition);
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
			
			
		Typical usage in HTML display context:
		
			$list = new DefinitionList("data/definitionlist.xml");
			if (! $list->parseFailed()) {
				while (($e = $list->getNextElement()) != null) {
					$e->printTerm();
					$e->printDefinition();
				}
			}
		
		
		XML Data Format:
		
			<definitionlist>
				<pagetitle>TEXT</pagetitle>
				<elements>
					<element>
						<term>TEXT</term>
						<definition>TEXT</definition>
					</element>
						.
						.
						.
				</elements>
			</definitionlist>
	*/
	require_once("XmlSimpleBaseClass.class.php");
	
	class DefinitionList extends XmlSimpleBaseClass {
		var $_element;
		var $xml_elements = array();
		var $_count = 0;
		
		function DefinitionList($filename) {
			$this->XmlSimpleBaseClass($filename);
			$this->_count = count($this->xml_elements);
		}
		
		function _p_start_element($parser, $element, $attributes) {	// was &$attributes
		 	parent::_p_start_element($parser, $element, $attributes);	// was &$attributes
		 	if ($element == "element") {
				$this->_element = new Element();
			}
		}
		
		function _p_cdata($parser, $text) {
			switch ($this->_tag) {
				case "term":
					$this->_element->xml_term .= $text;
					break;
				case "definition":
					$this->_element->xml_definition .= $text;
					break;
				case "pagetitle":
					$this->xml_pagetitle .= $text;
					break;
			}
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent,ENT_QUOTES,"ISO-8859-1"));
		}
		
		 function _p_end_element($parser, $element) {
		 	parent::_p_end_element($parser, $element);
		 	static $count = 0;
		 	if ($element == "element") {
				$this->_element->_index = $count;
				$this->xml_elements[$count++] = $this->_element;
			}
		 }
	
		function getNextElement() {
			if (list($key, $element) = each($this->xml_elements)) {
				return $element;
			}
			else {
				reset($this->xml_elements);
				return null;
			}
		}
		
		function getElement($elementid) {
			return $this->xml_elements[$elementid];
		}
		
		function addElement($term, $definition) {
			$element = new Element();
			$element->populate($term, $definition);
			$index = count($this->xml_elements);
			$element->_index = $index;
			$this->xml_elements[$index] = $element;
			ksort($this->xml_elements);
			$this->save();
		}
		
		function updateElement($elementid, $term, $definition) {
			$this->xml_elements[$elementid]->populate($term, $definition);
			$this->save();
		}
		
		function deleteElement($elementid) {
			unset($this->xml_elements[$elementid]);
			ksort($this->xml_elements);
			$this->save();
		}
		
		function promoteElement($elementid) {
			if ($elementid > 0 && $elementid < count($this->xml_elements)) {
				$curr = $this->xml_elements[$elementid];
				$this->xml_elements[$elementid] = $this->xml_elements[$elementid-1];
				$this->xml_elements[$elementid-1] = $curr;
				$this->save();
			}
		}
		
		function demoteElement($elementid) {
			if ($elementid >= 0 && $elementid < count($this->xml_elements)-1) {
				$curr = $this->xml_elements[$elementid];
				$this->xml_elements[$elementid] = $this->xml_elements[$elementid+1];
				$this->xml_elements[$elementid+1] = $curr;
				$this->save();
			}
		}
	}
	
	class Element {
		var $xml_term;
		var $xml_definition;
		var $_index;
		
		function Element() {
			;
		}
		
		function populate($term, $definition) {
			$this->xml_term = $term;
			$this->xml_definition = $definition;
		}
		
		function printID() {
			print htmlentities($this->_index,ENT_QUOTES,"ISO-8859-1");
		}
		
		function printTerm() {
			print htmlentities($this->xml_term,ENT_QUOTES,"ISO-8859-1");
		}
		
		function printDefinition() {
			print $this->htmlFilterSelective($this->xml_definition);
		}
		
		function printDefinitionForEdit() {
			print htmlentities($this->xml_definition,ENT_QUOTES,"ISO-8859-1");
		}
	}
?>
