<?	// copyright (C) 2005 - Matt Newberry, matt@mattnewberry.net
	// all rights reserved, except as licensed under contract
	
	/*  A Calendar is a container for Event objects; each Event contains an expiration date, 
		a display date (which may be any suitable text) and a description. Events are ordered ascending 
		by date code.
	
		Public Interface:
		
				//-- constructor ---------------
			
			function Calendar($filename)
				- creates a Calendar from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function printPageTitle();
				- prints an identifying string, html-encoded, for an instance of this class (from XmlSimpleBaseClass)
			
			function getNextEvent();
				- returns the next Event sequentially; returns null if no more Events
			
			function Event.printID();
				- prints the Event's identifier; mainly for use as a url parameter
			
			function Event.printDate();
				- prints the Event's display date, html-encoded
			
			function Event.printDescription();
				- prints the Event's description, html-encoded
			
			function Event.printDescriptionTruncated($maxlen);
				- prints the Event's description truncated to $maxlen chars, html-encoded
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the Calendar was successfully parsed from xml, 
					otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function printParseMessage();
				- prints error message from the PHP xml parser, html-encoded (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function getEvent($key);
				- returns an Event by it's internal identifier; may return null
			
			function addEvent($expires, $date, $description);
				- add an Event in order by expiration date
			
			function deleteEvent($key);
				- delete the Event with the given key
				
			function monthName($monthnum);
				- return printable month name for $monthnum between 1 and 12, otherwise empty string
			
			function Event.setExpiration($exp);
				- set the Event's expiration date, which must be a legal date
			
			function Event.setDisplayDate($date);
				- append text to the Event's display date, which can be any meaningful string
			
			function Event.setDescription($desc);
				- append text to the Event's description
			
			function Event.populate($expires, $date, $description);
				- set all Event fields to new values
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
		
		
		Typical usage in HTML display context:
		
			$calendar = new Calendar("data/calendar.xml");
			if (! $calendar->parseFailed()) {
				while (($e = $calendar->getNextEvent()) != null) {
					$e->printDate();
					$e->printDescription();
				}
			}
		
		
		XML Data Format:
		
			<calendar>
				<pagetitle>TEXT</pagetitle>
				<events>
					<event>
						<expires>INTEGER-DATE-CODE</expires>
						<displaydate>TEXT</displaydate>
						<description>TEXT</description>
					</event>
						.
						.
				</events>
			</calendar>
	*/
	require_once("XmlSimpleBaseClass.class.php");
	
	class Calendar extends XmlSimpleBaseClass {
		var $_event;
		var $xml_events = array();
		var $_count = 0;
		var $_months = array("January","February","March","April","May","June",
								"July","August","September","October","November","December");
		
		function Calendar($filename) {
			$this->XmlSimpleBaseClass($filename);
			$this->_count = count($this->xml_events);
		}
		
		 function _p_start_element($parser, $element, $attributes) {	// was &$attributes
		 	parent::_p_start_element($parser, $element, $attributes);	// was &$attributes
		 	if ($element == "event") {
				$this->_event = new Event();
			}
		 }
		
		function _p_cdata($parser, $text) {
			switch ($this->_tag) {
				case "expires":
					$this->_event->xml_expires .= $text;
					break;
				case "displaydate":
					$this->_event->xml_displaydate .= $text;
					break;
				case "description":
					$this->_event->xml_description .= $text;
					break;
				case "pagetitle":
					$this->xml_pagetitle .= $text;
					break;
			}
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent,ENT_QUOTES,"ISO-8859-1"));
		}
		
		 function _p_end_element($parser, $element) {
		 	parent::_p_end_element($parser, $element);
		 	if ($element == "event") {
				if ($this->_event->xml_expires > time())		// filter expired events
					$this->xml_events[$this->_event->xml_expires] = $this->_event;
			}
		 	else if ($element == "events") {
				ksort($this->xml_events);
			}
		 }
		
		function getNextEvent() {
			if (list($key, $event) = each($this->xml_events)) {
				return $event;
			}
			else {
				reset($this->xml_events);
				return null;
			}
		}
		
		function getEvent($key) {
			return $this->xml_events[$key];
		}
		
		function addEvent($expires, $date, $description) {
			$event = new Event();
			while (array_key_exists($expires, $this->xml_events))
				++$expires;
			$event->populate($expires, $date, $description);
			$this->xml_events[$expires] = $event;
			ksort($this->xml_events);
			$this->save();
		}
		
		function updateEvent($key, $expires, $date, $description) {
			$this->deleteEvent($key);
			$this->addEvent($expires, $date, $description);
			$this->save();
		}
		
		function deleteEvent($key) {
			foreach ($this->xml_events as $e) {
				if ($e->xml_expires == $key)
					unset($this->xml_events[$key]);
			}
			ksort($this->xml_events);
			$this->save();
		}
		
		function monthName($monthnum) {
			if ($monthnum > 0 && $monthnum <= 12)
				return $this->_months[$monthnum-1];
			else
				return "";
		}
	}
	
	class Event {
		var $_linkRegex = '{\\b((https?://|mailto:)([\\w/\\#:.?+=&%@!\\-]+?))(?=[.:?\\-]*(?:[^\\w/\\#:.?+=&%@!\\-]|$))}';	// http, https, mailto
		var $_anchorRegex = '/(@A\[)([^\s]+)(\s)([^\]]+)(\])/i';
		var $_relativelinkRegex = '/(@R\[)([^\s]+)(\s)([^\]]+)(\])/i';
		var $_httpRegex = '/(@H\[)([^\s]+)(\s)([^\]]+)(\])/i';
		var $_mailtoRegex= '/(@M\[)([^\s]+)(\s)([^\]]+)(\])/i';
		var $_imageRegex= '/(@G\[)([^\s]+)(\s)([^\]]+)(\])/i';
		var $_fontRegex = '{@B\[([^\]]+)\]|@BI\[([^\]]+)\]|@I\[([^\]]+)\]}i';	// bold | bold-italic | italic
		var $xml_expires;
		var $xml_displaydate;
		var $xml_description;
		
		function Event() {
			;
		}
		
		function populate($exp, $date, $desc) {
			$this->xml_expires = $exp;
			$this->xml_displaydate = $date;
			$this->xml_description = $desc;
		}
		
		function printID() {
			print htmlentities($this->xml_expires,ENT_QUOTES,"ISO-8859-1");
		}
		
		function printDate() {
			print htmlentities($this->xml_displaydate,ENT_QUOTES,"ISO-8859-1");
		}
		
		function printDescription() {
			print $this->htmlFilterSelective($this->xml_description);
		}
		
		function printDescriptionForEdit() {
			print htmlentities($this->xml_description,ENT_QUOTES,"ISO-8859-1");
		}
		
		function printDescriptionTruncated($maxlen) {
			print $this->htmlFilterSelective(substr($this->xml_description, 0, $maxlen));
			if (strlen($this->xml_description) > $maxlen)
				print "...";
		}
		
		function printDescriptionTruncatedForEdit($maxlen) {
			print htmlentities(substr($this->xml_description, 0, $maxlen),ENT_QUOTES,"ISO-8859-1");
			if (strlen($this->xml_description) > $maxlen)
				print "...";
		}
		
		function htmlFilterSelective($text) {
			$output = htmlentities($text,ENT_QUOTES,"ISO-8859-1");
			$output = preg_replace($this->_linkRegex, "<a href=\"$1\">$3</a>", $output);
			$output = preg_replace($this->_anchorRegex, "<a name=\"$2\">$4</a>", $output);
			$output = preg_replace($this->_relativelinkRegex, "<a href=\"$2\">$4</a>", $output);
			$output = preg_replace($this->_httpRegex, "<a href=\"http://$2\">$4</a>", $output);
			$output = preg_replace($this->_mailtoRegex, "<a href=\"mailto:$2\">$4</a>", $output);
			$output = preg_replace($this->_imageRegex, "<img src=\"$2\" alt=\"$4\">", $output);
			$output = preg_replace($this->_fontRegex, "<strong>$1<em>$2</em></strong><em>$3</em>", $output);
			return $output;
		}
	}
?>
