<?	// copyright (C) 2005 - Matt Newberry, matt@mattnewberry.net
	// ALL RIGHTS RESERVED, except as site-licensed under contract
	
	/*  A Menu is a container for MenuItem objects each MenuItem contains a label and a url.  The ordering of 
		menu items is controlled by the administrator.
	
		Public Interface:
		
				//-- constructor ---------------
			
			function Menu($filename)
				- creates a Menu from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
				
			function getNextMenuItem();
				- returns the next MenuItem sequentially; returns null if no more MenuItems
				
			function MenuItem.printID();
				- prints the MenuItem's unique id
				
			function MenuItem.printLabel();
				- prints the MenuItem's text, html-encoded
			
			function MenuItem.printUrl();
				- prints the MenuItem's url, html-encoded
			
			function MenuItem.printMenuItem();
				- prints the MenuItem, html-encoded, as a hyperlink
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the Menu was successfully parsed from xml, otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function printParseMessage();
				- prints error message from the PHP xml parser, html-encoded (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function getMenuItem($menuitemid);
			
			function addMenuItem($label, $url);
			
			function updateMenuItem($menuitemid, $label, $url);
			
			function deleteMenuItem($menuitemid);
			
			function promoteMenuItem($menuitemid);
				- move the item up one place in the display order
			
			function demoteMenuItem($menuitemid);
				- move the item down one place in the display order
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
			
			
		Typical usage in HTML display context:
		
			$menu = new Menu("data/menu.xml");
			if (! $menu->parseFailed()) {
				while (($i = $menu->getNextMenuItem()) != null) {
					<a href="$i->printUrl()">$i->printLabel()</a>;
								or
					$i->printMenuItem();
				}
			}
		
		
		XML Data Format:
		
			<menu>
				<menuitems>
					<menuitem>
						<label>TEXT</label>
						<url>TEXT</url>
					</menuitem>
						.
						.
						.
				</menuitems>
			</menu>
	*/
	//require_once("../XmlSimpleBaseClass.class.php");

	class Menu extends XmlSimpleBaseClass {
		var $xml_menuitems = array();		// indexed
		var $_menuitem;
		
		function Menu($filename) {
			$this->XmlSimpleBaseClass($filename);
		}
		
		 function _p_start_element($parser, $element, &$attributes) {
		 	parent::_p_start_element($parser, $element, &$attributes);
		 	if ($element == "menuitem") {
				$this->_menuitem = new MenuItem();
			}
		 }
		
		function _p_cdata($parser, $text) {
			switch ($this->_tag) {
				case "label":
					$this->_menuitem->xml_label .= $text;
					break;
				case "url":
					$this->_menuitem->xml_url .= $text;
					break;
			}
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent));
		}
		
		 function _p_end_element($parser, $element) {
		 	parent::_p_end_element($parser, $element);
		 	static $count = 0;
		 	if ($element == "menuitem") {
				$this->_menuitem->_index = $count;
				$this->xml_menuitems[$count++] = $this->_menuitem;
			}
		 }
		
		function getNextMenuItem() {
			if ($menuitem = current($this->xml_menuitems)) {
				next($this->xml_menuitems);
				return $menuitem;
			}
			else {
				reset($this->xml_menuitems);
				return false;
			}
		}
		
		function addMenuItem($label, $url) {
			$menuitem = new MenuItem();
			if ($this->_s_cstrpos(strtolower($url), "http://") != 0)
				$url = "http://".$url;
			$menuitem->populate($label, $url);
			$this->xml_menuitems[] = $menuitem;
			$this->save();
		}
		
		function updateMenuItem($menuitemid, $label, $url) {
			$menuitem =& $this->getMenuItem($menuitemid);		// byref
			//if ($this->_s_cstrpos(strtolower($url), "http://") != 0)
			//	$url = "http://".$url;
			$menuitem->populate($label, $url);
			$this->save();
		}
		
		function getMenuItem($menuitemid) {
			return $this->xml_menuitems[$menuitemid];
		}
		
		function deleteMenuItem($menuitemid) {
			unset($this->xml_menuitems[$menuitemid]);	// TODO: check the indexes
			$this->save();
		}
		
		function promoteMenuItem($menuitemid) {
			if ($menuitemid > 0 && $menuitemid < count($this->xml_menuitems)) {
				$curr = $this->xml_menuitems[$menuitemid];
				$this->xml_menuitems[$menuitemid] = $this->xml_menuitems[$menuitemid-1];
				$this->xml_menuitems[$menuitemid-1] = $curr;
				$this->save();
			}
		}
		
		function demoteMenuItem($menuitemid) {
			if ($menuitemid >= 0 && $menuitemid < count($this->xml_menuitems)-1) {
				$curr = $this->xml_menuitems[$menuitemid];
				$this->xml_menuitems[$menuitemid] = $this->xml_menuitems[$menuitemid+1];
				$this->xml_menuitems[$menuitemid+1] = $curr;
				$this->save();
			}
		}
	}
	
		// ----- MenuItem class -----
	
	class MenuItem {
		var $xml_label;
		var $xml_url;
		var $_index;
		
		function MenuItem() {
			;
		}
		
		function populate($label, $url) {
			$this->xml_label = $label;
			$this->xml_url = $url;
		}
		
		function printID() {
			print htmlspecialchars($this->_index);
		}
		
		function printLabel() {
			print htmlspecialchars($this->xml_label);
		}
		
		function printUrl() {
			print htmlspecialchars($this->xml_url);
		}
		
		function printMenuItem() {
			print "<a href=\"".htmlspecialchars($this->xml_url)."\">".htmlspecialchars($this->xml_label)."</a>";
		}
	}
	
?>
