#!/usr/local/bin/perl
#
# Rensselaer Polytechnic Institute's mail preener.
# 
# $Id: preenmail.pl,v 1.30 2000/04/17 13:33:54 sofkam Exp $
#
# Copyright (c) 1992-1995 by Rensselaer Polytechnic Institute
# All rights reserved.
# 
# Permission is hereby granted, without written agreement and without
# license or royalty fees, to use, copy, modify, and distribute this
# software and its documentation for any purpose, provided that the
# above copyright notice and the following two paragraphs appear in
# all copies of this software.
#  
# IN NO EVENT SHALL RENSSELAER POLYTECHNIC INSTITUTE BE LIABLE TO ANY
# PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
# DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION,
# EVEN IF RENSSELAER POLYTECHNIC INSTITUTE HAS BEEN ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
# 
# RENSSELAER POLYTECHNIC INSTITUTE SPECIFICALLY DISCLAIMS ANY
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE
# PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND RENSSELAER POLYTECHNIC
# INSTITUTE HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
# ENHANCEMENTS, OR MODIFICATIONS.
# 
# Version 2.0 created by Sanjay Dec 92.
# Revised 17 Feb 1993
#
# Heavily revised beginning Sept 1996 -- mds.
#
# Collect header and message separately to avoid accidental
# matches in forwarded messages.  Report on savings in verbose
# mode.  From line pattern adjusted for new mail server. --mds.
# 
# Took out -remove option---it was missleading.  Now, by default
# the temporary files are removed, and there are two options to
# control this:  -preserve keeps the preened mail in the temp
# directory, and -test keeps both the preened mail and the new
# mailbox.  Catting these two together should re-create the original
# mailbox when mail is deleted (as opposed to expired), or -doit
# is not specified.
#
# Renamed -w to -doit to be more clear about its meaning.
#
# Preserved mail is now given the name <mbox>.preened.  This will
# allow preenmail to preserv mail between runs.  More complex preservation
# schemes will require a script.  --mds 25 October 1996.
#
# Modified for the new mail machine:
#
#    Preened mail location, library include path, check for
#    poplock file, -stats, -log and -compress options.
#
#    Lots of additional error checking and recovery (e.g.,
#    closing down present mbox and proceeding to next).
#
#                   --mds 6-12 December 1996.
#
# Added -clean option to clean old preened mail from the temp dir.
#
#                   --mds 23 January 1997
#
# Added -daysread option.  Replaced multiple DAEMON messages with
# a single message.  Cleaned up some code.
#
#                   --mds 14 January 1998.
#
# More code cleanup.  Changed $pattern to $PATTERN so it is clear this
# is a global.  Added checks for $MESSAGE_MATCH (was $MESSAGE_BAD)
# and $MESSAGE_EXPIRE to ProcessMessage().  Got rid of redundant
# $DeleteMessage flag ($MESSAGE_STATUS was enough).
#
# Changed -delete option to -summary option, because what it really did
# was print a summary to the users mailbox.
#
# Added -daysexpired option to remove mail from expired user mail boxes sooner
# than regular user mail boxes.
#
# Clarified usage messages and verbose reporting.
#
#                    --mds 15 January 1998.
#
# Keep summary information by message type.  Keep track of message sizes,
# instead of stating file.  Added -summarystats option.  Fixed error
# in summary message.  If -agesexpired was specified, 21 days was always
# reported, even if 120 was the criterion.  The actual delete decision
# was correct.
#
#                    --mds 22 January 1998.
#
# addNDays was doing an incorrect year calculation.  If $year was
# less than 100 it returned 1900+$year, otherwise it returned $year.
# After 1/1/2000 this would result in a returned $year of 100.
#
# Added some more deeper levels of debugging, and a -verboseall option
# to set the verbosity leve to max.
#
#		-- mds 3 January 2000.
#
# Added a -bodymatch <pattern> option to match a pattern in the body
# of a message, and a -virus flag to indicate the message is being
# matched/removed because it matches a virus pattern.
#
#		--mds 12 January 2000.
#
# Added -expunge option and removed -expire option
# Added feature to prevent removal of first message with X-IMAP or X-IMAPBase
# Added parameter to control $MailSpool and $PopLockDir
#
#               --jwh 28 September 2008 

# To Do:
#
#    Include a ``will be expired soon'' message.
#
#    -summarystats could use more descriptive statistics.
#

require 5.003;
require Fcntl;
use Fcntl;
require "timelocal.pl";		# perl library for time routine.

$OverWrite = 0;                 # Really modify mailbox.
$Purge = 0;                     # Purge the mailbox, don't attempt to keep preened mail.
$FileSpecified = 0;             # A specific file was specified.

$RemoveTempFiles = 1;		# Remove both temp files by default.
$RemovePreenedMail = 1;		# Remove the preened mail by default.

$DoAge = 0;			# "Age" (expire/delete) mail over $DefaultNumDays days old.
$DoAgeExpired = 0;              # "Age" expiring accounts after $DefaultNumDaysExpired.
$DoAgeRead = 0;                 # "Age" read mail over $ExpireReadNumDays days old.
$DoMatch = 0;                   # "Age" mail matching pattern
$DoBodyMatch = 0;		# "Age" mail with body matching pattern
$VirusMatch = 0;		# The PATTERN matches an email virus.
$VirusName = "";		# The name of the virus.
$DoExpunge = 0;			# Expunge mail marked for deletion (X-Status: D).
$Summary = 0;   		# Don't append summary message by default.

$DefaultNumDays = 120;		# Default experation for unread mail.
$DefaultNumDaysRead = 21;       # Default experation for read mail.
$DefaultNumDaysExpired = 21;    # If the account is expiring, mail kept for 21 days.

$X_EXPIRE_DAYS = 30;		# No longer used. - jwh 9/28/08

$NumDays = $DefaultNumDays;
$NumDaysExpired = $DefaultNumDaysExpired;
$NumDaysRead = $DefaultNumDaysRead;

# LOCAL: Fill out the location of the mail spool and the poplock files.
#        These are the root directories, ignoring any hashing.
#
# Can be specified in parameters now 9/28/08 - jwh
#
$MailSpool  = "/var/spool/mail"; # The mail spool root directory.
$PopLockDir = "/var/spool/mail"; # The location of pop lock files.

# This is the default file/dir of files (-file option).
$DefaultDir = "/var/spool/mail";

$TempDir = "/var/spool/preen";
$Hashed = 0;                    # The directory is hashed.
$HashedPOP = 0;                 # The poplock directory is also hashed.
$MailLock = 0;                  # Use System V maillock() semantics.

# Templates for various temporary files.
$TempFileTemplate = "$TempDir/preen.";
$PreenMboxTemplate = "$TempDir/preened.";

$Verbose = 0;
$Stats = 0;			# Print a single line stats message.
$SummaryStats = 0;              # Stats for entire run.
$Log = 0;			# Send stats line to local0.info.
$Compress = 0;			# Compress the preserved mail.
$Clean = 0;			# Clean old preserved mail.
$FilesChecked = 0;		# How many files have been checked so far?

$MESSAGE_OK = 0;                # The message should not be expired or expunged.
$MESSAGE_EXPUNGED = 1;          # The message is marked for deletion (X-Status: D) 
$MESSAGE_OLD = 2;               # The message is old and can be expired/deleted.
$MESSAGE_MATCH = 3;             # The message header matched a search pattern.
$MESSAGE_READ = 4;              # The message has been read and can be expired/deleted.
$MESSAGE_EXOLD = 5;             # The account is expiring and message is old.
$MESSAGE_VIRUS = 6;		# The message contains a virus.
$MESSAGE_VIRUS_READ = 7;        # The message contains a virus, and is status RO.
$MESSAGE_DO_NOT_REMOVE = 8;     # The message contains X-IMAP or X-IMAPBase Header
$MESSAGE_STATUS = $MESSAGE_OK;  # Current status of the message.

$MessagesDeleted = 0;		# Number of messages deleted from mbox.
$MessagesProcessed = 0;         # Number of messages processed.

# An array to keep track of message types.
@MessageActions = (0, 0, 0, 0, 0, 0, 0);
@MessageSavings = (0, 0, 0, 0, 0, 0, 0);

# For -summarystats
$MBoxesChecked = 0;     # Total number of mboxes checked.
$MBoxesPreened = 0;     # Total number of mboxes preened.
$MBoxesEmpty   = 0;     # Total 0 byte mboxes.
$UnPreenedOK   = 0;     # Total messages in non-preened mboxes.
$UnPreenedBytes= 0;      # Total bytes in non-preened mboxes.

#Accumulating individual message stats for preened messages.
@SummaryMessageActions = (0, 0, 0, 0, 0, 0, 0);
@SummaryMessageSavings = (0, 0, 0, 0, 0, 0, 0);

#Indexes into the above arrays.
($OK, $OLD, $EXOLD, $READ, $EXPUNGED, $MATCH, $ERROR) = 0..6;

umask 0177;

# Get an exclusive lock on the $fd file descriptor.  $fd
# must be opened for writing.

# Commented out to shutup perl -cw.
# $LOCK_SH = 1;
$LOCK_EX = 2;
# $LOCK_NB = 4;
$LOCK_UN = 8;

sub lock (*) {
    my $fd = shift;
    flock($fd, $LOCK_EX);
}

# Unlock the file descriptor.
sub unlock (*) {
    my $fd = shift;
    flock($fd, $LOCK_UN);
}


# Create and lock the pop lock file.  Only make one attempt
# to lock file file.  If it failes, we'll preen it next time.

sub lockfile ($) {
    my $rcsid = shift();

    if ($HashedPOP) {
	my @chars = split //, $rcsid;
	my $LOCKFILE = "$PopLockDir" . "/%s/%s/.%s.pop";
	return sprintf $LOCKFILE, $chars[0], $chars[1], $rcsid;
    } else {
	my $LOCKFILE = "$PopLockDir" . "/.%s.pop";
	return sprintf $LOCKFILE, $rcsid;
    }
}

# Create and lock the pop lock file.  The parameter "$user"
# is used to construct the pop lock path, and is the name
# of the mail box.

sub pop_lock ($) {
    my $user = shift();
    my $handle = "$user" . "_pop";
    my $poplock = lockfile $user;
    my $status;

    $status = sysopen($handle,  $poplock , O_RDWR | O_CREAT | O_EXCL , 0600);

    # Success.
    if ($status) {
	lock ($handle);
	chown ((getpwnam($user))[2], (getgrnam("mail"))[2], $poplock);
    }

    return $status;
}

# Remove the pop lock file.
sub pop_unlock ($){
    my $user = shift();
    my $handle = "$user" . "_pop";
    my $poplock = lockfile $user;

    unlink($poplock);
    unlock($handle);
    close($handle);
}


# Lock the mail box.  Depending on your system, this may involve
# creating a <userid>.lock file.

# For systems that need to create a lockfile for the mail drop.
sub maildrop_lockfile ($) {
    my $rcsid = shift();

    if ($Hashed) {
	my @chars = split //, $rcsid;
	my $LOCKFILE = "$MailSpool" . "/%s/%s/%s.lock";
	return sprintf $LOCKFILE, $chars[0], $chars[1], $rcsid;
    } else {

	my $LOCKFILE = "$MailSpool" . "/%s.lock";
	return sprintf $LOCKFILE, $rcsid;
    }
}

sub maildrop_lock ($) {
    my $user = shift();
    my $handle = "$user" . "_lock";
    my $maillock = maildrop_lockfile $user;
    my $stutus;

    $status = sysopen($handle,  $maillock , O_RDWR | O_CREAT | O_EXCL , 0600);

    # Success.
    if ($status) {
	lock ($handle);
	chown ((getpwnam($user))[2], (getgrnam("mail"))[2], $maillock);
    }

    return $status;
}

# Remove the pop lock file.
sub maildrop_unlock ($){
    my $user = shift();
    my $handle = "$user" . "_lock";
    my $maillock = maildrop_lockfile $user;

    unlink($maillock);
    unlock($handle);
    close($handle);
}


# Print an error message to STDERR, and optionally
# send it to logger.

sub print_err ($) {
    my $message = shift();

    print STDERR "ERR $message\n";
    if ($Log) {
	system ("/usr/bin/logger", "-i", "-plocal0.info", "-tpreenmail", "ERR $message");
    } 
}


#
# Use associative arrays for days and months.
#
%days = (  Sun, 0,
	   Mon, 1,
	   Tue, 2,
	   Wed, 3,
	   Thu, 4,
	   Fri, 5,
	   Sat, 6);

%months = (  Jan, 0,
	   Feb, 1,
	   Mar, 2,
	   Apr, 3,
	   May, 4,
	   Jun, 5,
	   Jul, 6,
	   Aug, 7,
	   Sep, 8,
	   Oct, 9,
	   Nov, 10,
	   Dec, 11);

#
# Specify usage of preenmail. All options can be shortened.
#

$usage = "preenmail [-days [<n>]] [-daysread [<n>] [-daysexpired] [-expunge]
          [-summary] [-match <pattern)>] [-bodymatch <pattern>] [-virus <name>]
          [-maillock] [-file <file/dir>...] [-hashed] [-hashedpop] [-tempdir <dir>]
          [-mailspool <dir>] [-poplockdir <dir>] [-test] [-preserve] [-compress]
          [-clean] [-verbose...|-verboseall] [-stats] [-log] [-purge] [-doit] [-help]

    -days [<n>]:        Expire mail older than N days (default $DefaultNumDays).
    -daysread [<n>]:    Expire read mail older than N days (default $DefaultNumDaysRead).
    -daysexpired [<n>]: Expire from expiring accounts after N days (default $DefaultNumDaysExpired).
    -summary:             Send summary message to user.
    -expunge:             Expunge messages marked for deletion.
    -match <pattern>:     Delete mail whose header matches <pattern>.
    -bodymatch <pattern>: Delete mail whose header or body matches <pattern>.
    -virus <name>:        The <pattern> is the <name> virus (changes message).
    -file <file/dir>:     File or directory to examine (default $DefaultDir).
    -hashed:              The spool directory is hashed.
    -hashedpop:           The poplock directory is also hashed.
    -maillock:            Use System V maillock() file.
    -mailspool <dir>:     Use dir for System V maillock() files (default $MailSpool).
    -poplockdir <dir>:    Use dir for POP Lock files always (default $PopLockDir).
    -tempdir <dir>:       Use dir for temporary files (default $TempDir).
    -test:                Test mode keeps temp files in $TempDir.
    -preserve:            Preserve the preened mail in the tempdir.
    -purge:               Purge preened mail (not even a temp file).
    -compress:            Compress the preserved mail.
    -clean:               Clean old (> 7 days) preened mail out of the temp dir.
    -verbose:             Verbose mode (more == more verbose).
    -verboseall:          Very Verbose mode.
    -stats:               Prints status line for each mbox.
    -summarystats:        Prints summary statitics for run.
    -log:                 Send the status line to local0.info.
    -doit:                Actually do work (default is report process and report).
    -help:                Print this message.\n";
	
# Parse arguments
#
#

if ($#ARGV < 0) {
    print "$usage\n";
    exit 1;
}

while (@ARGV) {
    $_ = shift;
    if (/^-file$/) { 
		    $FileSpecified = 1; 
		    $File = shift; 
		    if ((!-f $File) && (!-d $File)) { 
			   die "Specify a file/directory name: -f \<filename\> \n"; }
		    next;
		   }
	elsif (/^-hashed$/) { $Hashed = 1; }
	elsif (/^-hashedpop$/) { $HashedPOP = 1; }
        elsif (/^-maillock$/) { $MailLock = 1; }
	elsif (/^-help$/) { print "$usage\n"; exit 1; }
        #
        # Remove messages matching a pattern in the header or body.
        #
	elsif (/^-match$/) { 
		    $DoMatch = 1; 
		    $PATTERN = shift;
		    if (-f $PATTERN) { 
			chop($PATTERN = `cat $PATTERN`); 
		    } elsif ($PATTERN =~ /^-(.*)/) { 
			die "Specify a pattern or filename with -match \n";
		    }
		    next; }
	elsif (/^-bodymatch$/) { 
		    $DoBodyMatch = 1; 
		    $PATTERN = shift;
		    if (-f $PATTERN) { 
			chop($PATTERN = `cat $PATTERN`); 
		    } elsif ($PATTERN =~ /^-(.*)/) { 
			die "Specify a pattern or filename with -bodymatch \n";
		    }
		    next; }
	elsif (/^-virus$/) {
	            $VirusMatch = 1;
		    $VirusName = shift;
		    next; }
        #
        # These must be read carefully, so that the optional argument
        # is set correctly.
        #
	elsif (/^-days$/) { 
			$DoAge = 1; 	
			$NumDays = shift; 
			if (($NumDays =~ /^\D+$/) || ($NumDays <= 0)) {
			    unshift (@ARGV, $NumDays) if ($NumDays ne "");
			    $NumDays = $DefaultNumDays; 
			}
			next; }
	elsif (/^-daysread$/) { 
			$DoAgeRead = 1; 	
			$NumDaysRead = shift; 
			if (($NumDaysRead =~ /^\D+$/) || ($NumDaysRead <= 0)) {
			    unshift (@ARGV, $NumDaysRead) if ($NumDaysRead ne "");
			    $NumDaysRead = $DefaultNumDaysRead; 
			}
			next; }
	elsif (/^-daysexpired$/) { 
			$DoAgeExpired = 1; 	
			$NumDaysExpired = shift; 
			if (($NumDaysExpired =~ /^\D+$/) || ($NumDaysExpired <= 0)) {
			    unshift (@ARGV, $NumDaysExpired) if ($NumDaysExpired ne "");
			    $NumDaysExpired = $DefaultNumDaysExpired; 
			}
			next; }

        elsif (/^-summary$/) { $Summary = 1; next; }
	elsif (/^-expunge$/) { $DoExpunge = 1; next; }
	elsif (/^-test$/) {  $RemoveTempFiles = 0; next; }
        elsif (/^-preserve$/) { $RemovePreenedMail = 0; next; }
        elsif (/^-purge$/) { $Purge = 1; next; }
        elsif (/^-compress$/) { $Compress = 1; next; }
        elsif (/^-clean$/)    { $Clean = 1; next; }
	elsif (/^-tempdir$/) { $TempDir = shift; 
		            $TempFileTemplate = "$TempDir/preen.";
                            $PreenMboxTemplate = "$TempDir/preened.";
			    next; }
	elsif (/^-mailspool$/) { $MailSpool = shift; }
	elsif (/^-poplockdir$/) { $PopLockDir = shift; }
	elsif (/^-verbose$/) { $Verbose++; next; }
	elsif (/^-verboseall$/) { $Verbose = 65536; next; }
	elsif (/^-doit$/) { $OverWrite = 1; next; }
	elsif (/^-stats$/) { $Stats = 1; next; }
	elsif (/^-summarystats$/) { $SummaryStats = 1; next; }
	elsif (/^-log$/) { $Log = 1; next; }
	else {
		die "Unrecognized option: $_ \n Usage: $usage\n";
	}
}

#
# When an unread message is older that $NumDays (120 default), or a read
# message is older than $NumDaysRead, it removed by the program.  A summary
# of the message (who-from, how old, was it read, size) are printed to
# a message from the MAILER-DAEMON.  This message is than mailed to the
# user (appended to the mailbox) informing him/her of the DAEMON's actions.
#

#
# Checkile and see if the program is being run as root.
#

if (!-w "/") { die "You must be root to execute this program\n"; } 


# /* Main ``loop'' of Preenmail */

if ($Verbose) {
    print "preenmail ", `date '+%a %h %d %R %Y'`;

    if ($OverWrite) {
	print "Running in real mode.\n";
	print "Purging mail--not even keeping a temporary copy.\n" if $Purge;
    } else {
	print "Running in test mode.\n";
    }

    print "Expunge mail marked for deletion.\n" if $DoExpunge;
    print "Deleting Mail more then $NumDays old.\n" if $DoAge;
    print "Deleting Mail from expired accounts if more then $NumDaysExpired old.\n" if $DoAgeExpired;
    print "Deleting Read Mail more then $NumDaysRead old.\n" if $DoAgeRead;
    print "Deleting mail with pattern \"$PATTERN\".\n" if $DoMatch;
    print "Deleting mail with body pattern \"$PATTERN\".\n" if $DoBodyMatch;
    print "(The pattern is for the $VirusName virus)\n" if $VirusMatch;
}


if (!-d $TempDir) { 
    mkdir($TempDir, 0755);
}

# Process file, or directory of files.

if ( !$FileSpecified ) {
    $File = $DefaultDir;
}

if (-f $File ) {
    &check_file ($File);
} elsif (-d $File) {
    if ($Hashed) {
	foreach $key1 ('a'..'z') {
	    foreach $key2 ('a'..'z') {
		preen_dir("$File/$key1/$key2");
	    }
	}
    } else {
	preen_dir("$File");
    }

} else {
    print "Invalid file type :$File\n"; exit 1;
}

# Find and remove old (compressed) preened mail.
if ($Clean && $OverWrite) {
    if ($Verbose) { print "Cleaning old preened mail."; }
    open (PREENDIR, "/usr/bin/find $TempDir -mtime +7 -print|");
    while ($file = <PREENDIR>) {
	chop($file);
	if ($Verbose) { print "Unlinking $file\n"; }
	if (! -d $file) { unlink($file); }
    }
    close(PREENDIR);
}

if ($SummaryStats) {
    print "Summary Statisics for run: ", ($OverWrite ? "Resulted" : "Proposed"), "\n\n";

    print "Mail boxes Checked: $MBoxesChecked\n";
    print "Mail boxes empty:   $MBoxesEmpty\n";
    print "Mail boxes Preened: $MBoxesPreened\n";
    print "Mail boxes unpreened: ", $MBoxesChecked-$MBoxesPreened-$MBoxesEmpty, "\n";

    print "\nIn the Preened Mail Boxes:\n";
    print "\tOld Messages         = $SummaryMessageActions[$OLD] ($SummaryMessageSavings[$OLD])\n";
    print "\tOld Expired Messages = $SummaryMessageActions[$EXOLD] ($SummaryMessageSavings[$EXOLD])\n";
    print "\tRead Messages        = $SummaryMessageActions[$READ] ($SummaryMessageSavings[$READ])\n";
    print "\tMessages Marked for Deletion     = $SummaryMessageActions[$EXPUNGED] ($SummaryMessageSavings[$EXPUNGED])\n";
    print "\tMatch Messages       = $SummaryMessageActions[$MATCH] ($SummaryMessageSavings[$MATCH])\n";
    print "\tOK Messages          = $SummaryMessageActions[$OK] ($SummaryMessageSavings[$OK])\n";

    print "\nIn the UnPreened Mail Boxes:\n";
    print "\tOK Messages          = $UnPreenedOK ($UnPreenedBytes)\n";
    print "\n";
}

if ($Verbose) {
    print "preenmail ", `date '+%a %h %d %R %Y'`;
}


# End of ``Main Loop'' for preenmail.


# Check all of the files in $dir by calling check_file().

sub preen_dir($) {
    my $dir = shift();

    if ($Verbose) {
	    print "Opening dir :$dir\n";
    }

    if (-d $dir) {
	chdir ("$dir");
	opendir(DIR, ".");
	@files = grep(!/^\./, readdir(DIR));
	closedir(DIR);

	foreach $newfile (@files) {
	    &check_file ($newfile);
	}
    } else {
	print "Warning: $dir is not a directory\n"
    }
}

# Check an mbox file:
#
#    Loop and separate the file into a bunch of messages.
#
#        @MessageHeader is an array that contains the Message Header.
#
#        @Message is an array which will contain one full message and its header.
#
#    Operate on this array, and check if this message is to be expired, or
#    retained.
#
#               If retained, write message to $TempFile.
#               If expired, write the message to the $PreenMbox,
#                   and put summary in @PreenMessage.
#
#    If, at the end of the run, messages were preened then a new
#    summary message is appeneded to $TempFile.
	
sub check_file {
    local($File) = pop(@_);
    local($mbox);

    # Initialize Globals.
    $TempFile = $TempFileTemplate;
    $PreenMbox = $PreenMboxTemplate;

    $MESSAGE_STATUS = $MESSAGE_OK;
    $MessagesProcessed = 0;
    $MessagesDeleted = 0;

    # Actions taken on messages
    if (scalar(@MessageActions) > 0) { undef @MessageActions; }
    @MessageActions = (0, 0, 0, 0, 0, 0, 0);
    if (scalar(@MessageSavings) > 0) { undef @MessageSavings; }
    @MessageSavings = (0, 0, 0, 0, 0, 0, 0);

    # List of messages removed.
    if (scalar(@PreenMessage) > 0) { undef @PreenMessage; }


    if (!-f $File) { print_err("$File: is not a file"); return; }

    if ($Verbose) { print "Checking file : $File\n"; }
    $FilesChecked++;
    $0 = "preenmail[$FilesChecked]: $File";

    # File name is a global used for printing error messages.  It contains
    # either the mail box being checked, or the full path to the file being
    # checked.

    $FILENAME = $File;

    # mbox contains the mailbox name.  In the case of checking a directory
    # this is the same as $File, and $FILENAME.  In the case of checking
    # a specific file, it is just the mailbox without the path.

    if ($pos = rindex($File, "\/")) {
	$mbox = substr($File, ++$pos);
    } else {
	$mbox = $File;
    }

    # Open the file. If we are unable to open the file we return.
    # The file is opened for reading and writing so that it can be
    # LOCK_EX'ed.

    if (!open(FD, "+<$File")) {
	print_err("Could not open $File for rw: $!");
	return;
    }

    # Check for empty mailbox.
    if (eof(FD)) {
	if ($Verbose) { print "Empty file : $File\n"; }
	if ($Verbose || $Stats || $Log) {
	    if($SummaryStats) {
		$MBoxesChecked++;
		$MBoxesEmpty++;
	    }
	    print_stats ($mbox);
	}
	close(FD);
	return ;
    }

    # Check first line of mailbox and continue if it is good.
    $_ = <FD>;
    unless (/^From .*$/) {
	print_err("$File: Corrumpted mailbox.");
	close (FD);
	return;
    }

    # See if we can grab the pop lock file.  If not, then
    # return, someone is already reading the file---we'll
    # get it during the next preenmail run.

    if (!pop_lock($mbox)) {
	print_err("file: $mbox pop lock is busy.");
	close(FD);
	return;
    }

    # Now, lock the mbox file.  This lock will block, but we return
    # if it should somehow fail.
    if ($MailLock) { # System V maillock()
	if (!maildrop_lock($mbox)) {
	    print_err("file: $mbox lock is busy.");
	    close(FD);
	    return;
	}
    }
    if (!lock(FD)) {
	print_err("file: $mbox is busy.");
	close(FD);
	if ($MailLock) { maildrop_unlock($mbox); }
	pop_unlock($mbox);
	return;
    }

    # Construct temporary file names based on the mbox.

    $TempFile .= "$mbox" . "." . "$$" ;     # The temporary (possibly preeened) mbox.
    $PreenMbox .= "$mbox" . "." . "$$" ;    # The mail that was preened from mbox.
    $Recipient = $mbox;    # Used for sending mail notifying use of removed messags.

    # Open temp file, and a preened mailbox if actually modifying mailboxes.

    if (!open(TFD, ">$TempFile")) {
	print_err("cannot open $TempFile for writing: $!");
	unlock(FD);
	close(FD);
	if ($MailLock) { maildrop_unlock($mbox); }
	pop_unlock($mbox);
	return;
    }

    if ($OverWrite && !$Purge) {
    	if (!open(PFD, ">$PreenMbox")) {
	    print_err("cannot open $PreenMbox for writing: $!");
	    unlock(FD);
	    close(FD);
	    if ($MailLock) { maildrop_unlock($mbox); }
	    pop_unlock($mbox);
	    close(TFD);
	    unlink "$TempFile";
	    return;
	}
    }

    do {
	# Free previous messages (not that I don't trust
	# Perl's garbage collector or anything.
	if (scalar(@MessageHeader) > 0) { undef @MessageHeader; }
	if (scalar(@Message) > 0) { undef @Message; }

	$MessagesProcessed++;
	push(@MessageHeader, $_); # push most recent "From " line.

	if ($Verbose > 2) { print "Header : @MessageHeader"; }

	# Read the entire remainder of the header;
	until (/^\n$/ || eof(FD)) { push(@MessageHeader, $_ = <FD>); }

	# Read the entire message;
	@Message = @MessageHeader;
	until (/^From .*$/ || eof(FD)) { push(@Message, $_ = <FD>); }
	$_ = pop(@Message) if (!eof(FD)); # Pop "From " line of next message.

	# Process the message setting $MESSAGE_STATUS if a message
	# is too old, or read or matches the pattern, etc.
        #
        # Check for X-IMAP Headers first, so message will not be removed
        #

	$doesitmatch = 0;
	$doesitmatch = &match("^X-IMAP");
	if ($doesitmatch) 
             { $MESSAGE_STATUS = $MESSAGE_DO_NOT_REMOVE; }
	if ($DoMatch   && ($MESSAGE_STATUS == $MESSAGE_OK))
	     { &MatchPattern($PATTERN) ; }
	if ($DoBodyMatch   && ($MESSAGE_STATUS == $MESSAGE_OK))
	{
	    &MatchBodyPattern($PATTERN) ;
	    if ($VirusMatch && ($MESSAGE_STATUS == $MESSAGE_MATCH)) {
		$MESSAGE_STATUS = $MESSAGE_VIRUS;

		if (&CheckMessageRead($MessageHeader[1],
				      $MessageHeader[scalar(@MessageHeader)-2])) {
		    $MESSAGE_STATUS = $MESSAGE_VIRUS_READ;
		}
	    }
	}
	if ($DoExpunge  && ($MESSAGE_STATUS == $MESSAGE_OK))
	     { &CanExpire(); }
	if (($DoAge || $DoAgeExpired) && ($MESSAGE_STATUS == $MESSAGE_OK))
	{
	    $DaysOld = $NumDays;
	    $AccountExpired = 0;
	    if ($DoAgeExpired) {
		# Look up the $mbox as a passwd entry.
		($shell) = (getpwnam($mbox))[8];
		if ($shell eq "/bin/expired") {
		    $DaysOld = $NumDaysExpired;
		    $AccountExpired = 1;
		}
	    }

	    &HasExpired($DaysOld);

	    if (($MESSAGE_STATUS == $MESSAGE_OLD) && $AccountExpired) {
		$MESSAGE_STATUS = $MESSAGE_EXOLD;
	    }
	}
	if ($DoAgeRead && ($MESSAGE_STATUS == $MESSAGE_OK))
	     { &HasExpiredRead($NumDaysRead); }

	# Print message, record changes and clear flags.
	if (!(&ProcessMessage())) {
	    print_err("Problem writing $MessageHeader[0] from $File");
	    unlock(FD); close(FD);
	    if ($MailLock) { maildrop_unlock($mbox); }
	    pop_unlock($mbox);
	    &cleanup();
	    return ;
	}

    } while (!eof(FD));

    # If any messages were expired, write a summary to TFD.
    if ($MessagesDeleted && $Summary) {
	&ComposeMessage();
    }

    close (TFD);		# Close the temp file for writing.
    if ($OverWrite && !$Purge) { close (PFD);} # Close the preened mail.

    # Report on saving if any.  This must be done before overwriting
    # the old mailbox with the new mail box.
    if ($Verbose || $Stats || $Log) {
	&print_stats ($mbox);
    }

    # Gather summary stats for final report
    if ($SummaryStats) {
	$MBoxesChecked++;

	if ($MessagesDeleted) {
	    $MBoxesPreened++;

	    for ($i = $OK; $i <= $ERROR; $i++) {
		$SummaryMessageActions[$i] += $MessageActions[$i];
		$SummaryMessageSavings[$i] += $MessageSavings[$i];
	    }
	} else {
	    $UnPreenedOK += $MessageActions[$OK];
	    $UnPreenedBytes += $MessageSavings[$OK];
	}
    }

    # If any of the messages have been deleted and if we running it 
    # for real, we move the Tempfile over the original file.
    if ($MessagesDeleted && $OverWrite) {
	if (!open TFD, $TempFile) {
	    print_err("Connot open $TempFile for reading, overwrite aborted: $!");
	    unlock(FD);  close (FD);
	    if ($MailLock) { maildrop_unlock($mbox); }
	    pop_unlock($mbox);
	    &cleanup();
	    return;
	} else {
	    seek FD, 0, 0;
	    truncate FD, 0;
	    # HERE: This is a likely source of the core dump.
	    print FD <TFD>;
	    #       suggested fix:
	    # while (<TFD>) {
	    # 	  print FD $_;
	    # }
	    close TFD;
	}
    }

    &cleanup();    # Take care of temporary files and preened messages.
    unlock(FD);  close (FD);
    if ($MailLock) { maildrop_unlock($mbox); } # Remove .lock file after unlocking mailbox.
    pop_unlock($mbox);    # The pop file is unlocked after mbox is modified.
    
}

# Prints the summary line to logger and/or to stdout reporting on
# file savings by message type.

sub print_stats {
    my ($user) = @_;
    my $message;

    $message = sprintf("%s: %s %d %d %d %d %d %d %d %d %d %d %d %d\n",
		       ($OverWrite ? "Resulted" : "Proposed"), $user,
		       $MessageActions[$OLD], $MessageSavings[$OLD],
		       $MessageActions[$EXOLD], $MessageSavings[$EXOLD],
		       $MessageActions[$READ], $MessageSavings[$READ],
		       $MessageActions[$EXPUNGED], $MessageSavings[$EXPUNGED],
		       $MessageActions[$MATCH], $MessageSavings[$MATCH],
		       $MessageActions[$OK], $MessageSavings[$OK]);

    system ("/usr/bin/logger", "-i", "-plocal0.info", "-tpreenmail", "$message") if $Log;
    print $message if $Verbose || $Stats;
}

# If RemoveTempFiles, then remove the temp files created during the run.
# This is the default unless -test was specified.
#
# However, hang onto the preened (removed) mail if -preserve was
# specified (RemovePreenedMail is undefined).  Instead, rename the
# file to <mbox>.preened, or delete it anyway because it is 0 bytes
# long (no messages were removed).
#

sub cleanup {
    if ($RemoveTempFiles) {
	unlink ($TempFile);
	if ($OverWrite && !$Purge) {
	    if ($RemovePreenedMail || !($MessagesDeleted))	{
		unlink ($PreenMbox);
	    } else {
		($NewName = $PreenMbox) =~ s/preened\.(\w*)\.\d*/$1.preened/;
		if ($Verbose) { print "Preserving $NewName\n"; }
		rename $PreenMbox,  $NewName;
		if ($Compress) {
		    if ($Verbose) { print "Compressing $NewName\n"; }
		    system ("/usr/local/bin/gzip", "-f", "$NewName");
		}
	    }
	}
    }
}

#
#   ProcessMessage checks to to see if a message has been deleted. 
#
#   If the message has been deleted then it writes the message to 
#   the PreenMbox file, else it writes it to a TempFile.
#
#   If the status of the message is MESSAGE_OLD or MESSAGE_READ
#   it puts a summary into the @PreemMessage array.
#
#   If the message is properly handled, then ProcessMessage returns 1.
#   Otherwise, it will return 0.
# 

sub ProcessMessage {
    $message_size = get_message_size();
    if ($Verbose > 3) { print "message size = $message_size\n"; }

    if (($MESSAGE_STATUS == $MESSAGE_OK ) || ($MESSAGE_STATUS == $MESSAGE_DO_NOT_REMOVE )) {
	if ($Verbose > 3) { print "About to print message\n"; }
	if (! print TFD @Message) {
	    print_err("$FILENAME: Could not append to temporary file: $!");
	    return 0;
	}
	$MessageActions[$OK]++;
	$MessageSavings[$OK]+= $message_size;
    } else {
	# The message is expired, being expired, expunged, read or matched the
	# pattern record message status @PreenMessage and report actions.

	chop ($From = $MessageHeader[0]);

	if ($MESSAGE_STATUS == $MESSAGE_OLD) {
	    $Why = "Over $NumDays days old";
	    $MessageActions[$OLD]++;
	    $MessageSavings[$OLD]+= $message_size;
	} elsif ($MESSAGE_STATUS == $MESSAGE_EXOLD) {
	    $Why = "Over $NumDaysExpired days old and account expired";
	    $MessageActions[$EXOLD]++;
	    $MessageSavings[$EXOLD]+= $message_size;
	} elsif ($MESSAGE_STATUS == $MESSAGE_READ) {
	    $Why = "Read and over $NumDaysRead days old";
	    $MessageActions[$READ]++;
	    $MessageSavings[$READ]+= $message_size;
	} elsif ($MESSAGE_STATUS == $MESSAGE_EXPUNGED) {
	    $Why = "Marked for Deletion";
	    $MessageActions[$EXPUNGED]++;
	    $MessageSavings[$EXPUNGED]+= $message_size;
	} elsif ($MESSAGE_STATUS == $MESSAGE_MATCH) {
	    $Why = "Header or body matched $PATTERN";
	    $MessageActions[$MATCH]++;
	    $MessageSavings[$MATCH]+= $message_size;
	} elsif ($MESSAGE_STATUS == $MESSAGE_VIRUS) {
	    $Why = "Header or body matched virus pattern: $PATTERN
The message status is UNREAD, which means it does not appear to
have been downloaded to your email client.
The message has been preserved for later analysis\n";
	    $MessageActions[$MATCH]++;
	    $MessageSavings[$MATCH]+= $message_size;
	} elsif ($MESSAGE_STATUS == $MESSAGE_VIRUS_READ) {
	    $Why = "Header or body matched $VirusName virus pattern: $PATTERN
WARNING: The message status is READ, you might have a virus.  Please
update your virus scan software and check.  Don\'t panic! All this
means if you previously downloaded email which might contain a virus.
The message has been preserved for later analysis\n";
	    $MessageActions[$MATCH]++;
	    $MessageSavings[$MATCH]+= $message_size;
	} else {
	    # This should not happen.
	    $Why = "MESSAGE_STATUS == $MESSAGE_STATUS";
	    $MessageActions[$ERROR]++;
	    $MessageSavings[$ERROR]+= $message_size;
	}

	push(@PreenMessage, ">$From : $Why\n");
			
	if ($Verbose > 1) {
	    if (!$OverWrite) {
		print "$FILENAME: Would have deleted the following message:\n";
	    } else {
		print "$FILENAME: Deleting the following message:\n";
	    }
	    print "   $From : $Why\n";
	}

	if ($OverWrite && !$Purge)  {
	    if ($Verbose > 3) { print "About to print deleted message size\n"; }
	    if (! print PFD @Message) {
		print_err("Could not append to PMbox: $!");
		return 0;
	    }
	}
	$MessagesDeleted++; 
    }

    $MESSAGE_STATUS = $MESSAGE_OK ;
    return 1;
}

# Return the size of the @Message array.

sub get_message_size() {
    # $len = length(join("", @Message));
    my $len = 0;
    foreach $line (@Message) { $len += length($line); }
    return $len;
}

#
# MatchPattern looks for an exact match in the message header for a given expression.
#

sub MatchPattern {
    local($Pattern) = pop (@_);
    local($doesitmatch);

    $NumElements = @MessageHeader;
    if ($NumElements == 0) { return 1;}

    $doesitmatch = &match ($Pattern);
    if ($doesitmatch) {
	$MESSAGE_STATUS = $MESSAGE_MATCH;
    }
}

#
#   This sub-routine checks to see if a specified pattern is present 
#   in the message. If the pattern is "X-Status: D", then it also saves 
#   the line ( which is a header ) in an array. 
#
#   @ExpiredHeader not needed anymore - jwh 9/28/08
#   used subroutine to find X-IMAP and X-IMAPBase Headers
#   push X-IMAP Header in @IMAPHeader for future use
#

sub match {
    local($Pattern) = pop (@_);
    if (scalar(@IMAPHeader) > 0) { undef @IMAPHeader; }
    foreach $line (@MessageHeader) {
	if ($line =~ /^X-IMAP/) {
	     push(@IMAPHeader, $line); 
             $MESSAGE_STATUS = $MESSAGE_DO_NOT_REMOVE;
             }
	if ($line =~ /$Pattern/) 
	     { return 1;}
    }
    return 0;	
}

#
# MatchBodyPattern looks for an exact match in the message header or body for a given expression.
#

sub MatchBodyPattern {
    local($Pattern) = pop (@_);
    local($doesitmatch);

    $NumElements = @Message;
    if ($NumElements == 0) { return 1;}

    $doesitmatch = &matchbody ($Pattern);
    if ($doesitmatch) {
	$MESSAGE_STATUS = $MESSAGE_MATCH;
    }
}

sub matchbody {
    local($Pattern) = pop (@_);
    foreach $line (@Message) {
	if ($line =~ /$Pattern/) 
	     { return 1;}
    }
    return 0;	
}

#
#  Check if the current message is older than $days days old.
#

sub HasExpired {
    local($days) = pop(@_);    # how old before expiring.
    local($numelements);
    local($diff, $letter_time, $time_now);

    $numelements = @Message;   	# check for an empty message
    if ($numelements == 0) { return 1;}

    local (@Message_hdr) = @MessageHeader;
    if (&CheckHeaderFormat($Message_hdr[0])) {
	print_err("$FILENAME: line $.:Bad header: $Message_hdr[0]");
	return;
    }

    # Get the time from the "From " line.
    $letter_time = &get_time ($Message_hdr[0]);
    $time_now = time;
    $diff = $time_now - $letter_time;
    $threshold = 60 * 60 * 24 * $days;

    if ($Verbose > 4) {
	print "letter_time = $letter_time, time_now = $time_now, diff = $diff.\n";
	print "$diff ", ($diff > $threshold ? ">" : "<="),
	      " (60 * 60 * 24 * $days) = ", "$threshold\n";
    }

    if ($diff > $threshold) { 
	$MESSAGE_STATUS = $MESSAGE_OLD; 
    }
}

#
#  Check if the current message has been read,
#  and is older than $days days old.
#

sub HasExpiredRead {
    local($days) = pop(@_);    # how old before expiring.
    local($numelements);
    local($diff, $letter_time, $time_now);

    $numelements = @Message;   	# check for an empty message
    if ($numelements == 0) { return 1;}

    local (@Message_hdr) = @MessageHeader;
    if (&CheckHeaderFormat($Message_hdr[0])) {
	print_err("$FILENAME: line $.:Bad header: $Message_hdr[0]");
	return;
    }

    # Has the message has been read?
    if (&CheckMessageRead($Message_hdr[1], $Message_hdr[scalar(@Message_hdr)-2])) {

	# If yes, get the time from the "From " line.
	$letter_time = &get_time ($Message_hdr[0]);
	$time_now = time;
	$diff = $time_now - $letter_time;
	$threshold = 60*60*24*$days;

	if ($Verbose > 4) {
	    print "letter_time = $letter_time, time_now = $time_now, diff = $diff.\n";
	    print "$diff ", ($diff > $threshold ? ">" : "<="),
	          " (60 * 60 * 24 * $days) = ", "$threshold\n";
	}

	# is the message old enough to delete?
	if ($diff > $threshold) { 
	    $MESSAGE_STATUS = $MESSAGE_READ; 
	}
    }
}

#
#   Check if the message has an X-Status header and
#   if it has been marked for deletion.

sub CanExpire {
    local($gm_time, $doesitmatch);
    local($numelements);

    $numelements = @Message;
    if ($numelements == 0) { return 0;}

    $doesitmatch = &match("X-Status: D");
    if ($doesitmatch) {
	$MESSAGE_STATUS = $MESSAGE_EXPUNGED;
	#
	# Expunge messages immediately if marked for deletion. - jwh 9/28/08
	# This section is not needed. - jwh 9/28/08
	#
	# if (&CheckExpireDateFormat($ExpiredHeader[0])) { 
	#    print_err("$FILENAME: The expire header is not in the correct format");
	#    return ;
	#}
	#$gm_time = &get_gm_time;
	#if ( $gm_time == 1) { 	 # improper date format.
	#    $doesitmatch = 0; return 1;
	#}
	#$time_now = time;

	#if ($Verbose > 4) {
	#    print "gm_time = $gm_time",
	#          ($gm_time > $time_now ? ">" : "<="), "time_now = $time_now.\n";
	#}

	#if ( $time_now > $gm_time ) {
	#    $MESSAGE_STATUS = $MESSAGE_EXPIRED; 
	#}
	$doesitmatch = 0;
    }
}

# 
# The next two routines take a date string and return the time in seconds.
#

sub get_time {
    local($Pattern) = pop(@_);
    local($junk, $From, $day, $mon, $mday, $Time, $zone, $year);
    local($hour, $min, $sec, $timeofletter);
    ($junk, $From, $day, $mon, $mday, $Time, $zone, $year ) = split(' ', $Pattern);

    # Our current format doesn't have seconds, but the old one did.
    if ((@foo = split(/:/, $Time)) == 3) {
	($hour, $min, $sec) = @foo;
    } else {
	($hour, $min) = @foo;
	$sec = 0;
    }

    if (("$zone" ne "EST") && ("$zone" ne "EDT")) { $year = $zone ; }
    if ($year > 100) { $year -= 1900; }
    $timeofletter = &timelocal($sec, $min, $hour, $mday, $months{$mon}, $year, $days{$day});
    return $timeofletter;
}


#
# Subroutine not needed any more.  - jwh 9/28/08
#

sub get_gm_time {
    local($Pattern) = pop(@ExpiredHeader);
    local($junk, $mday, $mon, $year, $Time, $junk);
    local($hour, $min, $sec, $timeofletter);
    ($junk, $mday, $mon, $year, $Time, $junk) = split(' ', $Pattern);

    # Our current format doesn't have seconds, but the old one did.
    if ((@foo = split(/:/, $Time)) == 3) {
	($hour, $min, $sec) = @foo;
    } else {
	($hour, $min) = @foo;
	$sec = 0;
    }

    if ($year > 100) { $year -= 1900; }
    $timeofletter = &timelocal($sec, $min, $hour, $mday, $months{$mon}, $year, $days{$day});
    return $timeofletter;
}


#   ComposeMessage()
#
#   Compose a message from the MAILER-DAEMON explaining chages
#   to the mbox and options for common mailers.  This message
#   is appened to <TFD> ($TempFile), and has an expiration header
#   set for 30 days.
#
#   The summary of messages removed were placed in @PreenMessage
#   by ProcessMessage().
#

sub ComposeMessage {
    local ($ExpireDate, $CurrentDate, $Hostname, $DateField);
    # $ExpireDate and $X_EXPIRE_DAYS no longer used - jwh
    $ExpireDate = &AddNdays($X_EXPIRE_DAYS);
    chop ($CurrentDate = `date '+%a %h %d %R %Y'`);
    chop ($Hostname = `hostname`);
    chop ($DateField = `date -u '+%a, %d %h %Y %T GMT'`);
	
    my ($summary_message) = "From MAILER-DAEMON\@rpi.edu $CurrentDate
Received: by $Hostname, $CurrentDate
Date: $DateField
From: MAILER-DAEMON
To: $Recipient
Subject: Old mail deleted by MAILER-DAEMON.

The following messages have been removed from your mailbox on the mail
server either because they were marked for deletion, were over 
$NumDays days old, or they were already copied from the server to your
local machine and were over $NumDaysRead days old.  Or, the message
matched a pattern for known unsolicited mass mail or a virus.

You can remove old messages yourself (and avoid seeing these
MAILER-DAEMON messages) by setting your mail client options to delete
mail from the server after N days (where N is 1-21), or to not keep
mail on the server once it has been read.  Consult your manual/help
file, or send mail to hostmaster\@uuism.net for more information.

Message:                                                     Reason
------------------------------------------------------------------------
@PreenMessage

";

    my ($message_size) = length($summary_message);
    print "Printing summary message ($message_size bytes)\n" if $Verbose;

    # If all the processed messags are deleted, we don't need
    # the separator line.
    if ($MessagesDeleted != $MessagesProcessed) {
	print TFD "\n\n";
    }
    print TFD $summary_message;

    # Add the summary message to the count of ok messages.

    $MessageActions[$OK]++;
    $MessageSavings[$OK]+= $message_size;
}

#
# Check if the lines passed are:
#
#    X-UIDL: !*(&$*(!#@&$(*!#
#    Status: R
#
# Return 1 if true.

sub CheckMessageRead {
    chop ($UIDL = shift(@_));
    chop ($Status = shift(@_));

    # print "X-UIDL = $UIDL\n";
    # print "Status = $Status\n";

    if ( ( $UIDL =~ /^X-UIDL: .*$/ ) && ( $Status =~ /^Status:\s+RO\s*$/ ) ) {
	return 1;
    }

    return 0;
}

#
#  Sub routine to check the Date format for the X-RPI-Expires header
#  like : X-RPI-Expires: 21 Jan 1993 21:15:2 GMT
#
#  Subroutine Not needed anymore - jwh 9/28/08
#


sub CheckExpireDateFormat {
    chop ($Line = pop(@_)); 	# /* Get rid of newline */
    if ($Line =~ /^X-RPI-Expires: (\d+) (\w+) (\d+) (\d+):(\d+):(\d+) (\w+)$/) { 
		return 0; }
    else { 	
		return 1; }
}



#
#	/* Sub routine to check the format of the Message Header        */
#	/* like: From ramams@rpi.edu Wed Jun 16 22:04:31 1992 */
#	/*   or: From ramams@rpi.edu Wed Jun 16 22:04:31 EST 1992 */
#

# The format changed slightly from the SunOS mail machine to the
# solaris mail machine.  It is now something like:
#  
#  From sofkam Fri Sep 13 13:22 EDT 1996
#  
#  or
#  
#  From sofkam@rpi.edu Fri Sep 13 13:22 EDT 1996
#  
#  or maybe
#  
#  From sofkam@rpi.edu Fri Sep 13 13:22 1996
#
# Program did not like these Headers:
#
# ^From \S+                                    \s+\w+ \s+\w+ \s+\d+\s+\d+:\d+(:\d+)?\s+\w*   \s*\d+  $
# ^From sarcher@cedarlane.org@host.uuserver.net    Mon    Aug    11    09: 33 : 25               2008 -0500$
#
# What about trailing offset? added '( -\d+)?'  - jwh
#
# Program reported line number for next header:
#
# ^From \S+                               \s+\w+ \s+\w+ \s+\d+\s+\d+:\d+(:\d+)?\s+\w*   \s*\d+  $
# ^From prvs=Lsage=102591753@worldbank.org    Mon    Aug    11    12: 22 : 31               2008$


sub CheckHeaderFormat {
    chop ($Line = pop(@_)); 	# /* Get rid of newline */
#    if ($Line =~ /^From \S+\s+\w+\s+\w+\s+\d+\s+\d+:\d+(:\d+)?\s+\w*\s*\d+$/) {
     if ($Line =~ /^From \S+\s+\w+\s+\w+\s+\d+\s+\d+:\d+(:\d+)?\s+\w*\s*\d+( -\d+)?$/) {

	return 0;
    } else { 	
	return 1;
    }
}



#	The next sub routine generates a date N days from now.
#	This is used to generate an Expires: header. 
#

sub AddNdays {

        local ($Days) = pop( @_);
        local ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst);
        local ($Day, $Month, $Year, $FutureTime);
        $TimeNow = time;
	$FutureTime = $TimeNow + (60*60*24*$Days); # Add time in seconds.

	($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = gmtime($FutureTime);
	$Day = (Sun,Mon,Tue,Wed,Thu,Fri,Sat)[$wday];
	$Month = (Jan,Feb,Mar,Apr,May,Jun,Jul,Aug,Sep,Oct,Nov,Dec)[$mon];

        $Year = 1900 + $year;
	if ( $hour < 10) { $hour = "0" . "$hour"; }
	if ( $min < 10) { $min = "0" . "$min"; }
	if ( $sec < 10) { $sec = "0" . "$sec"; }
	return "$mday $Month $Year $hour:$min:$sec GMT";

}
