package DBFile;

#####################################################################
# Flat-file "Database" functionality
#####################################################################

# Create a new DBFile instance
# ----------------------------
sub new {
	my ($class,$table) = @_;
	my $self = bless { }, $class;
	$self->{'_dbfile'} = $table . ".txt";
	$self->{'_idfile'} = $table . ".id";
	$self->{'_propfile'} = $table . ".properties";
	$self->{'_lockfile'} = $table . ".lock";
	$self->{'_tempfile'} = $table . ".tmp";
	unless (-e $self->{'_dbfile'} && -w $self->{'_dbfile'}) {
		$self->fatalError("Database file [$self->{'_dbfile'}] does not exist or is not writeable!");
		}
	unless (-e $self->{'_idfile'} && -w $self->{'_idfile'}) {
		$self->fatalError("Database ID file [$self->{'_idfile'}] does not exist or is not writeable!");
		}
	unless (-e $self->{'_propfile'} && -w $self->{'_propfile'}) {
		$self->fatalError("Database Properties file [$self->{'_propfile'}] does not exist or is not writeable!");
		}
	unless (-e $self->{'_lockfile'} && -w $self->{'_lockfile'}) {
		$self->fatalError("Database Lock file [$self->{'_lockfile'}] does not exist or is not writeable!");
		}
	$self->{'_locked'} = 0;
	$self->{'_use_tmp_file'} = 0;
	$self->{'properties'} = {};
	$self->{'fieldorder'} = [];
	$self->{'fieldindex'} = {};
	$self->getFormat();
	$self->getFieldProperties();
	if (-e $self->{'_dbfile'}) {
		return $self;
		}
	else {
		return 0;
		}
	}

# Error-handling
# --------------
sub setErrorMessage {
	my ($self,$msg) = @_;
	$self->{'errorMessage'} = $msg;
	}
sub getErrorMessage {
	my ($self) = @_;
	return $self->{'errorMessage'};
	}
sub fatalError {
	my ($self,$msg) = @_;
	&main::FATALERROR($msg);
	}
sub fileError {
	my ($self,$msg, $file) = @_;
	$self->setErrorMessage($msg . $file);
	}
sub dataError {
	my ($self,$msg) = @_;
	$self->setErrorMessage($msg);
	}

# File-Locking routines
# ---------------------
sub lock {
	my ($self) = @_;
	if ($self->{'_locked'} > 0) {
		$self->{'_locked'}++;
		return $self->{'_locked'};
		}
	unless(open(LOCK,"> $self->{_lockfile}")) {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE",$self->{'_lockfile'}));
		return 0;
		}
	eval("flock(LOCK, 2)");
	$self->{'_locked'} = 1;
	return 1;
	}
sub unlock {
	my ($self) = @_;
	if ($self->{'_locked'} > 1) {
		$self->{'_locked'}--;
		return $self->{'_locked'};
		}
	eval("flock(LOCK, 8)");
	close(LOCK);
	$self->{'_locked'} = 0;
	return 1;
	}
sub copyTempToOriginal {
	my ($temp,$orig) = @_;
	require File::Copy;
	unless (File::Copy::move($temp,$orig)) {
		return $!;
		}
	return 0;
	}

#####################################################################
# DBMatch
#   Used to test values against column values in searches/updates
#####################################################################
sub DBMatch {
	my ($a,$b) = @_;
	# Regex matching
	if ($a =~ m|^/(.+)/$|) {
		my ($tmp) = $1;
		if ($b =~ /$tmp/i) { return 1; }
		else { return 0; }
		}
	# Less Than (=<)
	elsif ($a =~ m|^=<\s*(.+)$|) {
		if ($b < $1) { return 1; }
		else { return 0; }
		}
	# Greater Than (=>)
	elsif ($a =~ m|^=>\s*(.+)$|) {
		if ($b > $1) { return 1; }
		else { return 0; }
		}
	# Else string comparison
	else {
		if ($a eq $b) { return 1; }
		else { return 0; }
		}
	return 0;
	}

#####################################################################
# DBFormat
#   Make text into format ready for flat file
#####################################################################
sub DBFormat {
	my ($text) = "";
	$text = shift;
	$text =~ s|\r?\n|\\n|sg;
	$text =~ s|\r|\\n|sg;
	$text =~ s|\t|\\t|sg;
	return $text;
	}

#####################################################################
# DBUnformat
#   Un-format a string
#####################################################################
sub DBUnformat {
	my ($text) = "";
	$text = shift;
	$text =~ s|\\n|\n|sg;
	$text =~ s|\\t|\t|sg;
	return $text;
	}

#####################################################################
# makeRecord
#   Split a row into fields and return a record (hash ref)
#####################################################################
sub makeRecord {
	my ($self,$line) = @_;
	my (@values,$record);
	chomp($line);
	@values = split(/\t/,$line);
	unless ($#{$self->{'fieldorder'}} > 0) {
		$self->getFormat();
		}
	foreach $i (0 .. $#{$self->{'fieldorder'}}) {
		$record->{$self->{'fieldorder'}->[$i]} = &DBUnformat($values[$i]);
		}
	return $record;
	}

#####################################################################
# flattenRecord
#   Turn a record into a line of text
#####################################################################
sub flattenRecord {
	my ($self,$record) = @_;
	my (@values);
	unless ($#{$self->{'fieldorder'}} > 0) {
		$self->getFormat();
		}
	foreach $i (0 .. $#{$self->{'fieldorder'}}) {
		$fieldname = $self->{'fieldorder'}->[$i];
		push(@values,&DBFormat($record->{$fieldname}));
		}
	return join("\t",@values);
	}

#####################################################################
# getID
#   Get the ID counter for a DB File. Returns ID.
#####################################################################
sub getID {
	my ($self) = @_;
	my ($id);
	if (-e $self->{'_idfile'}) {
		if (open(ID,$self->{'_idfile'})) {
			$id = <ID>;
			chomp($id);
			close(ID);
			return $id;
			}
		else {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READ",$self->{'_idfile'}));
			return -1;
			}
		}
	else {
		if (open(ID,"> $self->{_idfile}")) {
			print ID "0";
			close(ID);
			return 0;
			}
		else {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE",$self->{'_idfile'}));
			return -1;
			}
		}
	}

#####################################################################
# incrementID
#   Increment the ID counter for a DB File
#####################################################################
sub incrementID {
	my ($self) = @_;
	my ($id);
	if (-e $self->{'_idfile'}) {
		$self->lock();
		if (open(ID,"+<$self->{_idfile}")) {
			$id = <ID>;
			chomp($id);
			$id++;
			seek(ID,0,0);
			print ID $id;
			close(ID);
			$self->unlock();
			return $id;
			}
		else {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READWRITE",$self->{'_idfile'}));
			$self->unlock();
			return 0;
			}
		}
	else {
		$self->lock();
		if (open(ID,"> $self->{_idfile}")) {
			print ID "0";
			close(ID);
			$self->unlock();
			return 0;
			}
		else {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE",$self->{'_idfile'}));
			$self->unlock();
			}
		}
	}

#####################################################################
# parseFormat
#   Parse a #format: line to define format of DB records
#####################################################################
sub parseFormat {
	my ($self,$format) = @_;
	my (@fields,$i);
	delete $self->{'fieldorder'};
	delete $self->{'fieldindex'};
	$format =~ s|^#format:\s*||;
	chomp($format);
	@fields = split(/\t/,$format);
	foreach $i (0 .. $#fields) {
		$self->{'fieldindex'}->{$fields[$i]} = $i;
		$self->{'fieldorder'}->[$i] = $fields[$i];
		}
	}

#####################################################################
# flattenFormat
#   Flatten the format fields into a #format: line
#####################################################################
sub flattenFormat {
	my ($self) = @_;
	return "#format:" . join("\t",@{$self->{'fieldorder'}});
	}

#####################################################################
# getFormat
#   Get the format of a DBFile
#####################################################################
sub getFormat {
	my ($self) = @_;
	my $format = "";
	if (open(DB,$self->{'_dbfile'})) {
		$format = <DB>;
		close(DB);
		chomp($format);
		$self->parseFormat($format);
		return 1;
		}
	else {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READ", $self->{'_dbfile'}));
		return 0;
		}
	}

#####################################################################
# getFieldProperties
#   Get the properties of database fields
#####################################################################
sub getFieldProperties {
	my ($self) = @_;
	unless (-e $self->{'_propfile'}) {
		return 1;
		}
	if (open(PROPERTIES,$self->{'_propfile'})) {
		while (<PROPERTIES>) {
			next if /^#/;
			next if /^\s*$/;
			chomp;
			s/\s*$//;
			($field,$value) = split(/=/);
			($name,$property) = split(/:/,$field);
			$self->{'properties'}->{$name}->{$property} = &DBUnformat($value);
			}
		close(PROPERTIES);
		return 1;
		}
	else {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READ",$self->{'propfile'}));
		return 0;
		}
	}

#####################################################################
# getRecords
#   Get records matching criteria. Return a reference to an array of records
#####################################################################
sub getRecords {
	my ($self,$criteria,$max_matches) = @_;
	my (@values,@matches,$line,$match,$i);
	my ($use_hash,$use_string);

	if ($max_matches eq "") { $max_matches = -1; }
	unless(open(DB,$self->{'_dbfile'})) {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READ", $self->{'_dbfile'}));
		return 0;
		}
	if (ref $criteria eq "HASH") { 
		$use_hash=1; $use_string=0; 
		}
	else { 
		$use_string=1; $use_hash=0; 
		$criteria =~ s|\$(\w+)|\$values[\$self->{'fieldindex'}->{'$1'}]|g;
		}
	while($line = <DB>) {
		next if $line=~/^#/;
		next unless ($line =~ /\S/);
		chomp($line);
		@values = split(/\t/,$line);
		$match = 1;
		if ($use_hash) {
			foreach $field (keys %{$criteria}) {
				$index = $self->{'fieldindex'}->{$field};
				$match = &DBMatch($criteria->{lc($field)} , $values[$index] );
				if ($match == 0) {
					last;
					}
				}
			}
		elsif ($use_string) {
			eval "unless ($criteria) { \$match=0; }";
			}
		if ($match == 1) {
			push(@matches,$line);
			if (($max_matches == 0) || (($max_matches != -1) && ($#matches+1 >= $max_matches)) ) {
				last;
				}
			}
		}
	close(DB);
	if ($#matches >= 0) {
		if ($max_matches == 1) {
			return $self->makeRecord($matches[0]);
			}
		else {
			foreach $i (0 .. $#matches) {
				$matches[$i] = $self->makeRecord($matches[$i]);
				}
			return \@matches;
			}
		}
	else {
		return 0;
		}
	}

#####################################################################
# getRecord
#   Get record matching criteria. Return only first match, if any.
#####################################################################
sub getRecord {
	my ($self,$properties) = @_;
	return $self->getRecords($properties,1);
	}

#####################################################################
# simpleSearch
#   Return all records having any item matching a simple regex
#####################################################################
sub simpleSearch {
	my ($self,$regex) = @_;
	my (@matches,$line,$i);
	$regex = quotemeta($regex);
	unless(open(DB,$self->{'_dbfile'})) {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READ", $self->{'_dbfile'}));
		return 0;
		}
	while($line = <DB>) {
		next if $line=~/^#/;
		next unless ($line =~ /\S/);
		next unless ($line =~ /$regex/io);
		push (@matches, $line);
		}
	close(DB);
	foreach $i (0 .. $#matches) {
		$matches[$i] = $self->makeRecord($matches[$i]);
		}
	return \@matches;
	}


#####################################################################
# addRecord
#   Add a record to a flat file
#####################################################################
sub addRecord {
	my ($self,$properties) = @_;
	my ($fields,$line);
	
	$self->lock();
	my ($id) = $self->incrementID();

	$properties->{'id'} = $id;
	$line = "";
	foreach $field (@{$self->{'fieldorder'}}) {
		if ($line ne "") {
			$line .= "\t";
			}
		if (exists $properties->{lc($field)}) {
			$line .= &DBFormat($properties->{lc($field)});
			}
		else {
			$line .= &DBFormat($self->{'properties'}->{lc($field)}->{'_db_default_value'});
			}
		}
	$line .= "\n";
	
	unless(open(DB,">> $self->{_dbfile}")) {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_APPEND",$self->{'_dbfile'}));
		return 0;
		}
	print DB $line;
	close(DB);
	$self->unlock();
	return $id;
	}

#####################################################################
# updateRecords
#   Updates records matching properties with new values
#   Returns the number of records changed
#####################################################################
sub updateRecords {
	my ($self,$properties,$newproperties,$delete) = @_;
	my ($fields,$line,@values,$count);
	my (@tmp);
	
	$self->lock();
	delete $newproperties->{'id'};
	unless(open(DB,$self->{'_dbfile'})) {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READ", $self->{'_dbfile'}));
		return 0;
		}
	if ($self->{'_use_tmp_file'}) { 
		unless(open(TMP,"> $self->{_tempfile}")) {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE", $self->{'_tempfile'}));
			return 0;
			}
		}
	while($line = <DB>) {
		next unless ($line =~ /\S/);
		if ($line=~/^#/) {
			if ($self->{'_use_tmp_file'}) { 
				print TMP $line;
				}
			else {
				push(@tmp,$line);
				}
			next;
			}
		chomp($line);
		@values = split(/\t/,$line);
		$match = 1;
		foreach $field (keys %{$properties}) {
			$index = $self->{'fieldindex'}->{$field};
			$match = &DBMatch($properties->{lc($field)} , $values[$index] );
			if ($match == 0) {
				last;
				}
			}
		if ($match == 1) {
			$count++;
			unless ($delete) {
				$record = $self->makeRecord($line);
				foreach (keys %{$newproperties}) {
					$record->{$_} = $newproperties->{$_};
					}
				if ($self->{'_use_tmp_file'}) { 
					print TMP $self->flattenRecord($record),"\n";
					}
				else {
					push(@tmp,$self->flattenRecord($record)."\n");
					}
				}
			}
		else {
			if ($self->{'_use_tmp_file'}) { 
				print TMP $line,"\n";
				}
			else {
				push(@tmp,$line."\n");
				}
			}
		}
	if ($self->{'_use_tmp_file'}) { 
		close(TMP);
		}
	close(DB);
	if ($self->{'_use_tmp_file'}) { 
		unless(my($err) = &copyTempToOriginal($self->{'_tempfile'}, $self->{'_dbfile'})) {
			$self->fileError(&main::getMessage("DBFILE_FILE_TEMP_COPY", $err, $self->{'_dbfile'}));
			return 0;
			}
		}
	else {
		unless(open(DB,"> $self->{_dbfile}")) {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE", $self->{'_dbfile'}));
			return 0;
			}
		foreach (@tmp) { print DB; }
		close(DB);
		}
	$self->unlock();
	#return $count;
	return 1;
	}

#####################################################################
# deleteRecords
#   Deletes records matching certain criteria
#####################################################################
sub deleteRecords {
	my ($self, $properties) = @_;
	return $self->updateRecords($properties, "", 1);
	}

#####################################################################
# writeFieldProperties
#   Write all field properties to properties file
#####################################################################
sub writeFieldProperties {
	my ($self) = @_;
	my ($field,$property);
	
	$self->lock();
	unless(open(PROP,"> $self->{_propfile}")) {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE", $self->{'_propfile'}));
		return 0;
		}
	foreach $field (keys %{$self->{'properties'}}) {
		foreach $property (keys %{$self->{'properties'}->{$field}}) {
			print PROP "$field:$property=$self->{properties}->{$field}->{$property}\n";
			}
		}
	close(PROP);
	$self->unlock();
	return 1;
	}

#####################################################################
# addField
#   Add a new field (column) to a DBFile. Default value accepted.
#####################################################################
sub addField {
	my ($self, $fieldname, $properties, $defaultvalue) = @_;
	my ($fields,$prop);
	my (@tmp);
	
	$self->lock();
	$fieldname =~ s/\W//g;
	$fieldname = lc($fieldname);
	
	if (exists $self->{'fieldindex'}->{$fieldname}) {
		$self->dataError(&main::getMessage("DBFILE_FIELD_ALREADY_EXISTS",$fieldname));
		return 0;
		}
	unless(open(DB,$self->{'_dbfile'})) {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READ",$self->{'_dbfile'}));
		return 0;
		}
	if ($self->{'_use_tmp_file'}) { 
		unless(open(TMP,"> $self->{_tempfile}")) {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE",$self->{'_tempfile'}));
			return 0;
			}
		}
	
	$self->{'fieldindex'}->{$fieldname} = $#{$self->{'fieldorder'}};
	push(@{$self->{'fieldorder'}}, $fieldname);
	
	while($line = <DB>) {
		chomp($line);
		next unless ($line =~ /\S/);
		if ($line =~ /^#format/) {
			$line .= "\t" . $fieldname;
			}
		else {
			$line .= "\t".&DBFormat($defaultvalue);
			}
		if ($self->{'_use_tmp_file'}) { 
			print TMP "$line\n";
			}
		else {
			push(@tmp,$line."\n");
			}
		}
	if ($self->{'_use_tmp_file'}) { 
		close(TMP);
		}
	close(DB);
	if ($self->{'_use_tmp_file'}) { 
		unless(my($err) = &copyTempToOriginal($self->{'_tempfile'}, $self->{'_dbfile'})) {
			$self->fileError(&main::getMessage("DBFILE_FILE_TEMP_COPY", $err, $self->{'_dbfile'}));
			return 0;
			}
		}
	else {
		unless(open(DB,"> $self->{_dbfile}")) {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE", $self->{'_dbfile'}));
			return 0;
			}
		foreach (@tmp) { print DB; }
		close(DB);
		}
	foreach (keys %{$properties}) {
		$self->{'properties'}->{$fieldname}->{$_} = $properties->{$_};
		}
	$self->writeFieldProperties();
	
	$self->unlock();
	return 1;
	}

#####################################################################
# editFieldProperties
#   Modify the properties of a field
#####################################################################
sub editFieldProperties {
	my ($self, $fieldname, $properties, $replace) = @_;
	my ($fields,$prop);

	$self->lock();
	$fieldname =~ s/\W//g;
	$fieldname = lc($fieldname);

	unless (exists $self->{'fieldindex'}->{$fieldname}) {
		$self->dataError(&main::getMessage("DBFILE_FIELD_DOESNT_EXIST",$fieldname));
		return 0;
		}
	if ($self->{'properties'}->{$fieldname}->{'_locked'}) {
		$self->dataError(&main::getMessage("DBFILE_FIELD_LOCKED",$fieldname));
		return 0;
		}
	if ($replace) {
		delete $self->{'properties'}->{$fieldname};
		}
	foreach (keys %{$properties}) {
		$self->{'properties'}->{$fieldname}->{$_} = $properties->{$_};
		}
	$self->writeFieldProperties();
	$self->unlock();
	return 1;
	}

#####################################################################
# replaceFieldProperties
#   Modify the properties of a field, wiping out any existing properties
#####################################################################
sub replaceFieldProperties {
	my ($self, $fieldname, $properties) = @_;
	return $self->editFieldProperties($fieldname, $properties, 1);
	}

#####################################################################
# renameField
#   Rename a field
#####################################################################
sub renameField {
	my ($self, $oldfieldname, $newfieldname) = @_;
	my ($line);
	my (@tmp);
	
	$self->lock();
	$fieldname =~ s/\W//g;
	$fieldname = lc($fieldname);

	unless (exists $self->{'fieldindex'}->{$oldfieldname}) {
		$self->dataError(&main::getMessage("DBFILE_FIELD_DOESNT_EXIST",$oldfieldname));
		return 0;
		}
	if (exists $self->{'fieldindex'}->{$newfieldname}) {
		$self->dataError(&main::getMessage("DBFILE_FIELD_ALREADY_EXISTS",$newfieldname));
		return 0;
		}
	if ($self->{'properties'}->{$oldfieldname}->{'_locked'}) {
		$self->dataError(&main::getMessage("DBFILE_FIELD_LOCKED",$oldfieldname));
		return 0;
		}
	unless(open(DB,$self->{'_dbfile'})) {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READ",$self->{'_dbfile'}));
		return 0;
		}
	if ($self->{'_use_tmp_file'}) { 
		unless(open(TMP,"> $self->{_tempfile}")) {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE",$self->{'_tempfile'}));
			return 0;
			}
		}
	
	while($line = <DB>) {
		chomp($line);
		next unless ($line =~ /\S/);
		if ($line =~ /^#format/) {
			$self->{'fieldindex'}->{$newfieldname} = $self->{'fieldindex'}->{$oldfieldname};
			delete $self->{'fieldindex'}->{$oldfieldname};
			grep { if ($_ eq $oldfieldname) { $_ = $newfieldname; } } @{$self->{'fieldorder'}};
			$line = $self->flattenFormat();
			}
		if ($self->{'_use_tmp_file'}) { 
			print TMP "$line\n";
			}
		else {
			push(@tmp,$line."\n");
			}
		}
	if ($self->{'_use_tmp_file'}) { 
		close(TMP);
		}
	close(DB);
	if ($self->{'_use_tmp_file'}) { 
		unless(my($err) = &copyTempToOriginal($self->{'_tempfile'}, $self->{'_dbfile'})) {
			$self->fileError(&main::getMessage("DBFILE_FILE_TEMP_COPY", $err, $self->{'_dbfile'}));
			return 0;
			}
		}
	else {
		unless(open(DB,"> $self->{_dbfile}")) {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE", $self->{'_dbfile'}));
			return 0;
			}
		foreach (@tmp) { print DB; }
		close(DB);
		}

	$self->{'properties'}->{$newfieldname} = $self->{'properties'}->{$oldfieldname};
	delete $self->{'properties'}->{$oldfieldname};
	$self->writeFieldProperties();

	$self->unlock();
	return 1;
	}

#####################################################################
# deleteField
#   Delete a field and all values in every row
#####################################################################
sub deleteField {
	my ($self, $fieldname) = @_;
	my ($line,$index,@values);

	$self->lock();
	$fieldname =~ s/\W//g;
	$fieldname = lc($fieldname);

	unless (exists $self->{'fieldindex'}->{$fieldname}) {
		$self->dataError(&main::getMessage("DBFILE_FIELD_DOESNT_EXIST",$fieldname));
		return 0;
		}
	if ($self->{'properties'}->{$fieldname}->{'_locked'}) {
		$self->dataError(&main::getMessage("DBFILE_FIELD_LOCKED",$fieldname));
		return 0;
		}
	unless(open(DB,$self->{'_dbfile'})) {
		$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_READ",$self->{'_dbfile'}));
		return 0;
		}
	if ($self->{'_use_tmp_file'}) { 
		unless(open(TMP,"> $self->{_tempfile}")) {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE",$self->{'_tempfile'}));
			return 0;
			}
		}
	
	$index = $self->{'fieldindex'}->{$fieldname};
	while($line = <DB>) {
		chomp($line);
		next unless ($line =~ /\S/);
		if ($line =~ /^#format/) {
			delete $self->{'fieldindex'}->{$fieldname};
			splice(@{$self->{'fieldorder'}}, $index, 1);
			$line = $self->flattenFormat();
			}
		elsif (/^#/) {
			# Do nothing to comment lines
			}
		else {
			@values = split(/\t/,$line);
			splice(@values, $index, 1);
			$line = join("\t",@values);
			}
		if ($self->{'_use_tmp_file'}) { 
			print TMP "$line\n";
			}
		else {
			push(@tmp,$line."\n");
			}
		}
	if ($self->{'_use_tmp_file'}) { 
		close(TMP);
		}
	close(DB);
	if ($self->{'_use_tmp_file'}) { 
		unless(my($err) = &copyTempToOriginal($self->{'_tempfile'}, $self->{'_dbfile'})) {
			$self->fileError(&main::getMessage("DBFILE_FILE_TEMP_COPY", $err, $self->{'_dbfile'}));
			return 0;
			}
		}
	else {
		unless(open(DB,"> $self->{_dbfile}")) {
			$self->fileError(&main::getMessage("DBFILE_FILE_OPEN_WRITE", $self->{'_dbfile'}));
			return 0;
			}
		foreach (@tmp) { print DB; }
		close(DB);
		}

	delete $self->{'properties'}->{$fieldname};
	$self->writeFieldProperties();

	$self->unlock();
	return 1;
	}

1;
